// Copyright 2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/12  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include "AccessBySerial.h"
#include "TSELogger.h"
#include <new>

/*
#define NOPARITY            0
#define ODDPARITY           1
#define EVENPARITY          2
#define MARKPARITY          3
#define SPACEPARITY         4

#define ONESTOPBIT          0
#define ONE5STOPBITS        1
#define TWOSTOPBITS         2
*/

AccessSerial::AccessSerial(const char *  tPortName, TSELogger& tLog): mLog(tLog)
{
	SerialClose();

	// CreateFile
	mPort = CreateFile(
		tPortName,
		GENERIC_READ | GENERIC_WRITE, 
		0, 
		NULL, 
		OPEN_EXISTING, 
		0, 
		NULL);

	if (mPort == INVALID_HANDLE_VALUE) {
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
		throw 10;
	}

	// Set parameters
	DCB dcb;
	if (!GetCommState(mPort, &dcb)) {
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
		SerialClose();
		throw 11;
	}

	// Basic settings
	dcb.BaudRate = 9600;
	dcb.ByteSize = 8;
	dcb.Parity = NOPARITY;						//NOPARITY ODDPARITY EVENPARITY
	dcb.StopBits = ONESTOPBIT;					//ONESTOPBIT ONE5STOPBITS TWOSTOPBITS ONESTOPBIT
	dcb.fParity = FALSE;

	// Special settings
	dcb.fOutxCtsFlow = FALSE;					// Disable CTS (Clear To Send)
	dcb.fOutxDsrFlow = TRUE;                    // Enable DSR (Data Set Ready)
	dcb.fOutX = FALSE;							// Disable XON/XOFF for transmission
	dcb.fInX = FALSE;							// Disable XON/XOFF for receiving
	dcb.fDtrControl = DTR_CONTROL_HANDSHAKE;    // Enable DTR (Data Terminal Ready)
	dcb.fRtsControl = RTS_CONTROL_DISABLE;      // Disable RTS (Ready To Send)
	dcb.XonChar = 0x11;							// Xon character
	dcb.XoffChar = 0x13;						// Xoff character
	if (!SetCommState(mPort, &dcb)) {
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
		SerialClose();
		throw 12;
	}

	// Set timeouts
	COMMTIMEOUTS timeouts;
	if (!GetCommTimeouts(mPort, &timeouts)) {
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
		SerialClose();
		throw 13;
	}
	timeouts.ReadIntervalTimeout = 20;
	timeouts.ReadTotalTimeoutMultiplier = 0; //10;
	timeouts.ReadTotalTimeoutConstant = 1000;
	timeouts.WriteTotalTimeoutMultiplier = 10;
	timeouts.WriteTotalTimeoutConstant = 1000;
	if (!SetCommTimeouts(mPort, &timeouts)) {
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
		SerialClose();
		throw 14;
	}
}

AccessSerial::~AccessSerial()
{
	SerialClose();
}

void AccessSerial::SerialSend(const std::vector<BYTE>& tBuffer)
{
	int nResult = 0;

	if (!SerialIsOpen())
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=NOT OPEN"));
		throw 20;
	}

	const BYTE * tempBuf;
	unsigned long remByteToSend = (unsigned long) tBuffer.size();
	unsigned long totalByteSent = 0;
	unsigned long bytesSent = 0;

	while (remByteToSend > 0)
	{
		tempBuf = tBuffer.data() + totalByteSent;
		nResult = WriteFile(mPort, (const char *)tempBuf, remByteToSend, &bytesSent, 0);

		if (nResult > 0)
		{
			remByteToSend -= bytesSent;
			totalByteSent += bytesSent;
		}
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
			throw 21;
		}//if nResult
	}
}

void AccessSerial::SerialRecv(std::vector<BYTE>& tBuffer, const unsigned long& tMaxSize, const int& tTimeout)
{
	unsigned long byteRcvd = 0;
	unsigned long byteRemain = tMaxSize;

	COMMTIMEOUTS timeouts;

	if (!SerialIsOpen())
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=NOT OPEN"));
		throw 30;
	}

	if (!GetCommTimeouts(mPort, &timeouts))
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
		throw 31;
	}

	tBuffer.clear();

	timeouts.ReadTotalTimeoutConstant = tTimeout;
	SetCommTimeouts(mPort, &timeouts);

	BYTE* rcvBuf = NULL;
	try {
		rcvBuf = new BYTE[tMaxSize];
	}
	catch (std::bad_alloc &) 
	{ 
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=ALLOC"));
		throw 32; 
	}

	while (byteRemain > 0)
	{
		if (ReadFile(mPort, rcvBuf, byteRemain, &byteRcvd, NULL))
		{
			if (byteRcvd > 0)
			{
				tBuffer.insert(tBuffer.end(), rcvBuf, rcvBuf + byteRcvd);
				byteRemain -= byteRcvd;
			}
			else
			{
				delete rcvBuf;
				return;// throw 33;
			}
		}
		else { //Timeout
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("serial error=%d", GetLastError()));
			delete rcvBuf;
			throw 34;
		}
	}
	delete rcvBuf;
}

BOOL AccessSerial::SerialIsOpen()
{
	if (mPort == INVALID_HANDLE_VALUE)
		return FALSE;
	return TRUE;
}

void AccessSerial::SerialClose()
{
	if (mPort == INVALID_HANDLE_VALUE)
		return;
	CloseHandle(mPort);
	mPort = INVALID_HANDLE_VALUE;
}

//EOF