/** Force Command Example 
 *  Sets the printer into "Force Command Mode" via ENPC (UDP)
 *  This allows the printer to accept TSE comamnds if the printer is in error mode.
 *  Such errors could be "out of paper" or "cover open" etc..
 */
#include "stdafx.h"
#include <stdio.h>
#include "ENPC_ForceSend.h"
using namespace std;

//#define HAFNIUM


/////////////////////////////////////////////////////////////////////////////
// Global variables
/////////////////////////////////////////////////////////////////////////////
#define MAX_TIMEOUT 	1	///< Max timeout seconds per one ENPC response
#define MAX_RETRY		3	///< Max retry count of ENPC
static int maxTimeout = MAX_TIMEOUT;
static int maxRetry = MAX_RETRY;

static DWORD dwError;		///< WSA Error

#define MAXBUF 1024
static BYTE sbuf[MAXBUF];	///< Send buffer for ENPC
static BYTE rbuf[MAXBUF];	///< Receive buffer for ENPC
static WORD wResult;		///< Result code of ENPC


/// 03-0015 Force sending command
static BYTE C03_0015[] = {
	'E', 'P', 'S', 'O', 'N',
	'C', 0x03, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x01,
	0x01,
};


/////////////////////////////////////////////////////////////////////////////

/** Create a socket descriptor
 * @note save the last error if create socket failed
 * @retval INVALID_SOCKET failed
 * @patval !INVALID_SOCKET success
 */
static SOCKET CreateSocket(void)
{
	// Create a socket
	SOCKET s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == INVALID_SOCKET)
		dwError = WSAGetLastError();
	return s;
}


/** Close a socket
 * @param s socket descriptor
 */
static int CloseSocket(SOCKET s)
{
	int n = closesocket(s);
	if (n == SOCKET_ERROR)
		dwError = WSAGetLastError();
	return n;
}



/** Send and receive ENPC command
 * @param s socket descriptor
 * @param ip IP address
 * @param len packet data length
 * @note both sbuf[] and rbuf[] buffer are used in this function
 * @retval 0 success
 * @retval <0 ENPC_ERROR
 */
static int SendRecvENPC(int s, ULONG ip, int len)
{
	int n;

	// Setting UDP port
	struct sockaddr_in addr = {0};
	addr.sin_family = AF_INET;
	addr.sin_port = htons(3289);
	addr.sin_addr.s_addr = ip;

	// Setting timeout
	struct timeval tv = {0};
	tv.tv_sec = maxTimeout;
	tv.tv_usec = 0;

	// Initialize retry counter
	int nRetry = 0;

SEND:
	// Send ENPC command
	n = sendto(s, (char*)sbuf, len, 0, (struct sockaddr*)&addr, sizeof(addr));
	if (n == SOCKET_ERROR) {
		dwError = WSAGetLastError();
		return ENPC_ERROR_SEND;
	}
	if (n < len)
		return ENPC_ERROR_SEND;

	while (1) {
		// Wait the ENPC response
		fd_set readfds;
		FD_ZERO(&readfds);
		FD_SET(s, &readfds);
		n = select(0, &readfds, NULL, NULL, &tv);
		if (n == 0) {
			if (++nRetry >= maxRetry)
				return ENPC_ERROR_TIMEOUT;
			goto SEND;
		}
		if (n == SOCKET_ERROR) {
			dwError = WSAGetLastError();
			return ENPC_ERROR_RECV;
		}

		// Receive ENPC response
		if (FD_ISSET(s, &readfds)) {
			struct sockaddr_in from = {0};
			int fromlen = sizeof(from);
			n = recvfrom(s, (char*)rbuf, MAXBUF, 0, (struct sockaddr*)&addr, &fromlen);
			if (n == SOCKET_ERROR) {
				dwError = WSAGetLastError();
				return ENPC_ERROR_RECV;
			}

			// Check the length of response
			if (n < 14)
				continue;

			// Check identifier of ENPC
			if (memcmp(&rbuf, "EPSON", 5) != 0)
				continue;

			// Check the type of ENPC packet
			switch (sbuf[5]) {
				case 'C':
					if (rbuf[5] != 'c')
						continue;
					break;
				default:
					continue;
					break;
				}

			// Check the function code of both send and receive packet
			if (memcmp(&sbuf[6], &rbuf[6], 4) != 0)
				continue;

			// Check that the printer actually entered force mode
			if (rbuf[14] != 0x01) {
				return ENPC_ERROR_OTHER;
			}

			// Save the result code of ENPC
			wResult = (rbuf[10] << 8) + rbuf[11];
			return n;
		}
	}
}


/////////////////////////////////////////////////////////////////////////////

/** Force sending
 * @param s socket
 * @param ip IP address
 * @param f flag
 * @retval 0 success
 * @retval <0 ENPC_ERROR
 */
static int ForceSend(SOCKET s, ULONG ip, BOOL f)
{
	int n;

	// Send ENPC command
	memcpy(sbuf, C03_0015, sizeof(C03_0015));
	sbuf[14] = f;
	n = SendRecvENPC((int)s, ip, sizeof(C03_0015));
	if (n < 0)
		return n;
	if (wResult != 0x0000)
		return ENPC_ERROR_RESULT;
	return 0;
}

/** Force sending
 * @param ip IP address
 * @param f flag
 * @retval 0 success
 * @retval <0 ENPC_ERROR
 */
int ENPC_ForceSend(ULONG ip, BOOL f)
{

	SOCKET s = CreateSocket();
	if (s == INVALID_SOCKET)
		return ENPC_ERROR_SOCKET;
	int n = ForceSend(s, ip, f);
	CloseSocket(s);
	return n;
}

