// Copyright 2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/12  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include "TSEAccessByEposDevice.h"
#include "AccessBySocket.h"
#include "TSESimpleJsonAndXmlParser.h"
#include "TSEUtil.h"
#include "TSELogger.h"
#include "TSEConfig.h"
#include <sstream>
#include <fstream>
#include <algorithm>

#define EPOS_DEVICE_PORT		"8009"
#define EPOS_FISCAL_DEVICEID	"local_TSE"
#define EPOS_FISCAL_DEVICETYPE	"type_storage"
#define EPOS_FISCAL_STORAGEMSG	"operate"

#define EPOS_CLIENTCONNECT_FILEEXT	".eposxmltseid"
#define DEFAULT_RCV_TIMEOUT		20000
#define LONGER_RCV_TIMEOUT		61000

#define TSECLIBDATAPATH				"\\EPSON\\TSE\\CLIB\\"


TSEAccessByEposDevice::TSEAccessByEposDevice(const std::string& tIpAddress, TSELogger& tLog):TSEAccess(tLog)
{
	unsigned long rcvTime = 0;
	std::vector<BYTE> recvBuffer;

	mASocket = NULL;
	mPort = EPOS_DEVICE_PORT;
	{
		TSEConfig tseConfig;
		std::string configXMLPort;
		if (tseConfig.GetEposDeviceXmlPort(configXMLPort) == true)
		{
			mPort = configXMLPort;
		}
	}
	
	try 
	{
		mASocket = new AccessSocket(tIpAddress.c_str(), mPort.c_str(), true, mLog);
	}
	catch (...)
	{
		mASocket = NULL;
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Create Socket Error"));
		throw std::runtime_error("");
	}

	try 
	{
		RecvMsg(recvBuffer, 0, DEFAULT_RCV_TIMEOUT, rcvTime);
	}
	catch (...)
	{
		delete mASocket;
		mASocket = NULL;
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Recv Error"));
		throw std::runtime_error("");
	}

	if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "connect")))
	{
		delete mASocket;
		mASocket = NULL;
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("XML (connect) data received from printer is Invalid"));
		throw std::runtime_error("");
	}

	mClientId = GetValueFromXml((char *)recvBuffer.data(), "client_id");
	mTseDevId = EPOS_FISCAL_DEVICEID;
	mLastDataId = "";
	mTseClibCompleteDataDir = "";
	mDeviceId = tIpAddress;
}

TSEAccessByEposDevice::~TSEAccessByEposDevice() 
{
	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd;	
	std::stringstream ssmsg;
	unsigned long rcvTime = 0;
	mTotalRcvTime = 0;
	bool checkDisconnect = true;

	ssmsg <<	"<disconnect>"
			<<		"<data>"
			<<			"<client_id>" << mClientId << "</client_id>"
			<<		"</data>"
			<<	"</disconnect>\0";

	std::string smsg = ssmsg.str();
	for (int i = 0; i < (int)smsg.size(); i++)
	{
	    sendCmd.push_back(*(smsg.c_str()+i));
	}
	sendCmd.push_back('\0');

	try 
	{
		SendMsg(sendCmd);
		RecvMsg(recvBuffer, 0, DEFAULT_RCV_TIMEOUT, rcvTime);
		mTotalRcvTime = rcvTime;
	}
	catch (int&) 
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Error during Disconnect"));
		checkDisconnect = false;
	}
	catch (...)
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Error during Disconnect. Other error"));
		checkDisconnect = false;
	}

	if (checkDisconnect)
	{
		if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "disconnect")))
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Error during Disconnect : %s", recvBuffer.data()));
		}
		if (GetValueFromXml((char *)recvBuffer.data(), "code") != "OK")
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Error during Disconnect : %s", recvBuffer.data()));
		}
	}

	delete mASocket;
	mASocket = NULL;
}


void TSEAccessByEposDevice::CreateConnectFile()
{
	std::string tseDevFileName = "";

	if (mTseClibCompleteDataDir == "")
	{
		std::string progDataPath = "";
		char szPath[4096];
		memset(szPath, '\0', 4096);
		std::string tDir = "";
		if (SUCCEEDED(GetEnvironmentVariable("ALLUSERSPROFILE", szPath, 4096)))
		{
			progDataPath = std::string(szPath);
		}
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Could not find  ALLUSERSPROFILE" ));
			return;
		}
		
		tDir = progDataPath + TSECLIBDATAPATH + "tmp";

		if (TSEUtil::CreateTseDirectory(tDir) == false)
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Failed to create Epson TSE data directory"));
			return;
		}

		mTseClibCompleteDataDir = progDataPath + TSECLIBDATAPATH;
	}

	tseDevFileName = mTseClibCompleteDataDir + GetConnectFileName();
	std::ofstream tseDevFileNew(tseDevFileName.c_str());
	if (tseDevFileNew.is_open())
	{
		tseDevFileNew << mClientId;
		tseDevFileNew.close();
	}

	return;
}

std::string TSEAccessByEposDevice::GetConnectFileName()
{
	std::string host = mDeviceId;
	std::string tseid = mTseDevId;
	std::transform(host.begin(), host.end(), host.begin(), toupper); 
	std::transform(tseid.begin(), tseid.end(), tseid.begin(), toupper);

	if ( (host == "127.0.0.1") || (host == "LOCALHOST") )
	{
		return "LOCALHOST_" + tseid + EPOS_CLIENTCONNECT_FILEEXT;
	}

	return host + "_" + tseid + EPOS_CLIENTCONNECT_FILEEXT;
}

void TSEAccessByEposDevice::SendRecv(const std::vector<BYTE>& sendCmd, std::vector<BYTE>& recvBuffer)
{
	unsigned long rcvTime = 0;

	mTotalRcvTime = 0;
	mTotalTime = 0;

	SYSTEMTIME st, et;
	union timeunion {
		FILETIME fT;
		ULARGE_INTEGER uL;
	} fst, fet;
	std::stringstream sLogAll;

	GetSystemTime(&st);
	SystemTimeToFileTime(&st, &fst.fT);

	try
	{
		SendMsg(sendCmd);
		RecvMsg(recvBuffer, 0, LONGER_RCV_TIMEOUT, rcvTime);
		mTotalRcvTime = rcvTime;
	}
	catch (int&)
	{
		if (Reconnect(true) == false)
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection interrupted and cannot reconnect."));
			throw std::runtime_error("");
		}
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection has been reconnected."));
			try
			{
				RecvMsg(recvBuffer, 0, DEFAULT_RCV_TIMEOUT, rcvTime); // Check if there are stream to receive
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Data received"));
				return;
			}
			catch (int&)
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Nothing to Receive."));
				throw std::runtime_error("");
			}
		}
	}
	catch (...)
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection Failed: Other error"));
		throw std::runtime_error("");
	}

	GetSystemTime(&et);
	SystemTimeToFileTime(&et, &fet.fT);
	mTotalTime = (unsigned long)((fet.uL.QuadPart - fst.uL.QuadPart) / 10000);

	return;
}


bool TSEAccessByEposDevice::EnableTseAccess()
{
	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd;
	std::stringstream ssmsg;
	std::string operationStatus = "";
	mTotalRcvTime = 0;
	mTotalTime = 0;
	
	ManagePreviousConnectionStatus();
	ssmsg <<	"<open_device>"
			<<		"<device_id>" << mTseDevId << "</device_id>"
			<<		"<data>"
			<<			"<type>" << EPOS_FISCAL_DEVICETYPE << "</type>"
			<<		"</data>"
			<<	"</open_device>\0";

	std::string smsg = ssmsg.str();
	for (int i = 0; i < (int)smsg.size(); i++)
	{
		sendCmd.push_back(*(smsg.c_str()+i));
	}
	sendCmd.push_back('\0');

	SendRecv(sendCmd, recvBuffer);

	if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "open_device")))
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("XML (open_device) data received from printer is Invalid."));
		throw std::runtime_error("");
	}

	operationStatus = GetValueFromXml((char *)recvBuffer.data(), "code");
	mLastDataId = GetValueFromXml((char *)recvBuffer.data(), "data_id");

	if (operationStatus != "OK")
	{
		return false;
	}

	CreateConnectFile();
	return true;
}

void TSEAccessByEposDevice::SendJsonStringToTse(const std::string& tJsonStr, std::string& tResponse)
{
	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd;
	std::stringstream ssmsg;
	std::string operationStatus = "";
	mTotalRcvTime = 0;
	mTotalTime = 0;
	
	ssmsg <<	"<device_data>"
			<<		"<device_id>" << mTseDevId << "</device_id>"
			<<		"<data>"
			<<			"<type>" << EPOS_FISCAL_STORAGEMSG << "</type>"
			<<			"<timeout>60000</timeout>"
			<<			"<requestdata>" << tJsonStr << "</requestdata>"
			<<		"</data>"
			<<	"</device_data>\0";

	std::string smsg = ssmsg.str();
	for (int i = 0; i < (int)smsg.size(); i++)
	{
	    sendCmd.push_back(*(smsg.c_str()+i));
	}
	sendCmd.push_back('\0');

	SendRecv(sendCmd, recvBuffer);

	if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "device_data")))
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("XML (device_data) data received from printer is Invalid"));
		throw std::runtime_error("");
	}

	operationStatus = GetValueFromXml((char *)recvBuffer.data(), "code");
	mLastDataId = GetValueFromXml((char *)recvBuffer.data(), "data_id");

	if (operationStatus != "SUCCESS")
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Executing 'device_data' on TSE not successful : [%s]", operationStatus.c_str()));
		throw std::runtime_error("");
	}
	tResponse = GetValueFromXml((char *)recvBuffer.data(), "resultdata");
}


bool TSEAccessByEposDevice::DisableTseAccess()
{
	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd;
	std::stringstream ssmsg;
	std::string operationStatus = "";
	mTotalRcvTime = 0;
	mTotalTime = 0;
	
	ssmsg <<	"<close_device>"
			<<		"<device_id>" << mTseDevId << "</device_id>"
			<<	"</close_device>\0";

	std::string smsg = ssmsg.str();
	for (int i = 0; i < (int)smsg.size(); i++)
	{
	    sendCmd.push_back(*(smsg.c_str()+i));
	}
	sendCmd.push_back('\0');

	SendRecv(sendCmd, recvBuffer);

	if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "close_device")))
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("XML (close_device) data received from printer is Invalid"));
		throw std::runtime_error("");
	}

	operationStatus = GetValueFromXml((char *)recvBuffer.data(), "code");
	mLastDataId = GetValueFromXml((char *)recvBuffer.data(), "data_id");

	if ((operationStatus != "OK") && (operationStatus != "DEVICE_NOT_OPEN"))
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("%s", operationStatus.c_str()));
		return false;
	}

	std::string tseDevFileName = "";
	if (mTseClibCompleteDataDir == "")
	{
		std::string progDataPath = "";
		char szPath[4096];
		memset(szPath, '\0', 4096);
		std::string tDir = "";
		if (SUCCEEDED(GetEnvironmentVariable("ALLUSERSPROFILE", szPath, 4096)))
		{
			progDataPath = std::string(szPath);
		}
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Could not find  ALLUSERSPROFILE"));
			return true;
		}

		tseDevFileName = progDataPath + TSECLIBDATAPATH + GetConnectFileName();
	}
	else
	{
		tseDevFileName = mTseClibCompleteDataDir + GetConnectFileName();
	}

	if (remove(tseDevFileName.c_str()) == 0)
	{
		//TSEPRINTLOG(LIBLOG_LEVEL_INFO, ("%s has been deleted", tseDevFileName.c_str()));
	}
	return true;
}


bool TSEAccessByEposDevice::Reconnect(bool doreconnect)
{
	int retry = 0;
	TSEPRINTLOG(mLog, LIBLOG_LEVEL_TRACE, ("Trying to Reconnect..."));
	while (true)
	{
		std::vector<BYTE> recvBuffer;
		unsigned long rcvTime = 0;
		std::string operationStatus = "";
		std::string newClientId = "";

		if (mASocket != NULL) delete mASocket;

		try 
		{
			mASocket = new AccessSocket(mDeviceId.c_str(), mPort.c_str(), true, mLog);
		}
		catch (int& )
		{
			mASocket = NULL;
			retry++;
			Sleep(1000);
			continue;
		}

		try
		{
			RecvMsg(recvBuffer, 0, DEFAULT_RCV_TIMEOUT, rcvTime);
		}
		catch (int&)
		{
			delete mASocket;
			mASocket = NULL;
			retry++;
			Sleep(1000);
			continue;
		}

		if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "connect")))
		{
			delete mASocket;
			mASocket = NULL;
			retry++;
			Sleep(1000);
			continue;
		}

		newClientId = GetValueFromXml((char *)recvBuffer.data(), "client_id");

		if (doreconnect)
		{
			std::vector<BYTE> sendCmd;
			std::stringstream ssmsg;
			
			ssmsg <<	"<reconnect>"
					<<		"<data>"
					<<			"<new_client_id>" << newClientId << "</new_client_id>"
					<<			"<old_client_id>" << mClientId << "</old_client_id>"
					<<			"<received_id>" << mLastDataId << "</received_id>"
					<<		"</data>"
					<<	"</reconnect>\0";

			std::string smsg = ssmsg.str();
			for (int i = 0; i < (int)smsg.size(); i++)
			{
		    	sendCmd.push_back(*(smsg.c_str()+i));
			}
			sendCmd.push_back('\0');

			try
			{
				SendMsg(sendCmd);
				RecvMsg(recvBuffer, 0, DEFAULT_RCV_TIMEOUT, rcvTime);
				mTotalRcvTime = rcvTime;
			}
			catch (int&)
			{
				return false;
			}

			if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "reconnect")))
			{
				return false;
			}
			operationStatus = GetValueFromXml((char *)recvBuffer.data(), "code");

			if (operationStatus == "OK")
			{
				mClientId = newClientId;
				ManagePreviousConnectionStatus();
				CreateConnectFile();
				return true;
			}

			return false;
		}
		else 
		{
			mClientId = newClientId;
			ManagePreviousConnectionStatus();
			CreateConnectFile();
			return true;
		}
		
		Sleep(1000);
		retry++;
	}
	return true;
}


bool TSEAccessByEposDevice::ManagePreviousConnectionStatus()
{
	std::string tseDevFileName = "";
	std::string prevClientId = "";

	if (mTseClibCompleteDataDir == "")
	{
		std::string progDataPath = "";
		char szPath[4096];
		memset(szPath, '\0', 4096);
		std::string tDir = "";
		if (SUCCEEDED(GetEnvironmentVariable("ALLUSERSPROFILE", szPath, 4096)))
		{
			progDataPath = std::string(szPath);
		}
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Could not find  ALLUSERSPROFILE"));
			return true;
		}

		tseDevFileName = progDataPath + TSECLIBDATAPATH + GetConnectFileName();
	}
	else
	{
		tseDevFileName = mTseClibCompleteDataDir + GetConnectFileName();
	}

	std::ifstream tseDevFile(tseDevFileName.c_str());
	if (tseDevFile.good()) 
	{
		if (tseDevFile.is_open())
		{
			getline(tseDevFile, prevClientId);
			tseDevFile.close();	
		}
	}

	if ((prevClientId != "") && (prevClientId != mClientId))
	{
		std::vector<BYTE> recvBuffer;
		std::vector<BYTE> sendCmd;
		std::stringstream ssmsg;
		unsigned long rcvTime = 0;
		mTotalRcvTime = 0;

		ssmsg <<	"<disconnect>"
				<<		"<data>"
				<<			"<client_id>" << prevClientId << "</client_id>"
				<<		"</data>"
				<<	"</disconnect>\0";

		std::string smsg = ssmsg.str();
		for (int i = 0; i < (int)smsg.size(); i++)
		{
		    sendCmd.push_back(*(smsg.c_str()+i));
		}
		sendCmd.push_back('\0');

		try 
		{
			SendMsg(sendCmd);
			RecvMsg(recvBuffer, 0, DEFAULT_RCV_TIMEOUT, rcvTime);
			mTotalRcvTime = rcvTime;
		}
		catch (int) {}
		catch (std::runtime_error &) {}

		if ((recvBuffer.empty()) || (!IsXmlResponseCorrect((char *)recvBuffer.data(), "disconnect")))
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Not a disconnect function"));
		}
		if (GetValueFromXml((char *)recvBuffer.data(), "code") != "OK")
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Error during Disconnect"));
		}
		
		if (remove(tseDevFileName.c_str()) == 0)
		{
			//TSEPRINTLOG(LIBLOG_LEVEL_INFO, ("%s has been deleted", tseDevFileName.c_str()));
		}
	}
	
	return true;
}

bool TSEAccessByEposDevice::IsXmlResponseCorrect(const char* tXmlString, const std::string& tTag)
{
	if (tXmlString == NULL)
	{
		return false;
	}
	return TseSimpleJsonAndXmlParser::XmlIsResponseCorrect(tXmlString, tTag);
}

std::string TSEAccessByEposDevice::GetValueFromXml(const char* tXmlString, const std::string& tTag)
{
	if (tXmlString == NULL)
	{
		return "";
	}
	return TseSimpleJsonAndXmlParser::XmlGetValue(tXmlString, tTag);
}

void TSEAccessByEposDevice::SendTo(const std::vector<unsigned char>& tBuffer)
{
	mASocket->SocketSend(tBuffer);
}

void TSEAccessByEposDevice::RecvFrom(std::vector<BYTE>& tBuffer, const unsigned long& tMaxSize, const unsigned long& tTimeout)
{
	mASocket->SocketRecv(tBuffer, tMaxSize, tTimeout);
}

//EOF