// Copyright 2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/12  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include "TSEAccessByEscPosNetwork.h"
#include "AccessBySocket.h"  
#include "TSELogger.h"
#include <sstream>

#define PRINT_PORT				"9100"
#define DEFAULT_RCV_TIMEOUT		20000
#define LONGER_RCV_TIMEOUT		60000

TSEAccessByEscPosNetwork::TSEAccessByEscPosNetwork(const std::string& tIpAddress, TSELogger& tLog) :TSEAccess(tLog)
{
	mASocket = NULL;
	mLog = tLog;

	try
	{
		mASocket = new AccessSocket(tIpAddress.c_str(), PRINT_PORT, false, mLog);
	}
	catch (...)
	{
		mASocket = NULL;
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Creating Socket Connection Failed."));
		throw std::runtime_error("");
	}

	mDeviceId = tIpAddress;
}

TSEAccessByEscPosNetwork::~TSEAccessByEscPosNetwork()
{
	delete mASocket;
	mASocket = NULL;
}

bool TSEAccessByEscPosNetwork::EnableTseAccess()
{
	unsigned long rcvTime = 0;

	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd{ 0x10, 0x14, 0x06, 0x01, 0x03, 0x01, 0x03, 0x14, 0x01, 0x06, 0x02, 0x08 };

	mTotalRcvTime = 0;
	mTotalTime = 0;

	SYSTEMTIME st, et;
	union timeunion {
		FILETIME fT;
		ULARGE_INTEGER uL;
	} fst, fet;
	std::stringstream sLogAll;
	GetSystemTime(&st);
	SystemTimeToFileTime(&st, &fst.fT);

	try
	{
		SendMsg(sendCmd);
		while (true)
		{
			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Enabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x37) continue;
			mTotalRcvTime += rcvTime;
			
			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Enabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x5C) continue;
			mTotalRcvTime += rcvTime;
			break;
		}

		RecvMsg(recvBuffer, 2, DEFAULT_RCV_TIMEOUT, rcvTime);
		mTotalRcvTime += rcvTime;
	}
	catch (int&) 
	{
		if (Reconnect(true) == false) 
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection Error: Connection interrupted and cannot reconnect."));
		}
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection interrupted but reconnected."));
		}

		throw std::runtime_error("");
	}
	catch (...) 
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection Error: Other error."));
		throw std::runtime_error("");
	}

	if (recvBuffer[0] != 0x30)
	{
		return false;
	}

	GetSystemTime(&et);
	SystemTimeToFileTime(&et, &fet.fT);
	mTotalTime = (unsigned long)((fet.uL.QuadPart - fst.uL.QuadPart) / 10000);

	return true;
}

void TSEAccessByEscPosNetwork::SendJsonStringToTse(const std::string& tJsonStr, std::string& tResponse)
{
	unsigned long rcvTime = 0;

	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd{ 0x1C, 0x28, 0x50, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00 };
	std::vector<BYTE> sendAck{ 0x06 };
	std::vector<BYTE> sendCmdJson;

	unsigned long jsonSize = (unsigned long) tJsonStr.length();

	sendCmd[3] = (jsonSize + 4) & 0xFF;				//set pL
	sendCmd[4] = ((jsonSize + 4) >> 8) & 0xFF;		//set pH
	sendCmdJson.insert(sendCmdJson.end(), sendCmd.data(), sendCmd.data() + sendCmd.size());
	sendCmdJson.insert(sendCmdJson.end(), tJsonStr.c_str(), tJsonStr.c_str() + jsonSize);

	tResponse = "";
	mTotalRcvTime = 0;
	mTotalTime = 0;

	SYSTEMTIME st, et;
	union timeunion {
		FILETIME fT;
		ULARGE_INTEGER uL;
	} fst, fet;
	std::stringstream sLogAll;
	GetSystemTime(&st);
	SystemTimeToFileTime(&st, &fst.fT);

	try
	{
		SendMsg(sendCmdJson);
		BYTE status = 0;
		do
		{
			while(true)
			{
				RecvMsg(recvBuffer, 1, LONGER_RCV_TIMEOUT, rcvTime);
				if (recvBuffer.size() != 1)
				{
					TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
					throw std::runtime_error("");
				}
				if (recvBuffer[0] != 0x53) continue;
				mTotalRcvTime += rcvTime;

				RecvMsg(recvBuffer, 1, LONGER_RCV_TIMEOUT, rcvTime);
				if (recvBuffer.size() != 1)
				{
					TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
					throw std::runtime_error("");
				}
				if (recvBuffer[0] != 0x27) continue;
				mTotalRcvTime += rcvTime;
				break;
			}
			RecvMsg(recvBuffer, 4, LONGER_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 4)
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
				throw std::runtime_error("");
			}

			mTotalRcvTime += rcvTime;
			status = recvBuffer[0];
			unsigned short bsize = (((unsigned short)recvBuffer[3]) << 8) | recvBuffer[2];

			RecvMsg(recvBuffer, bsize, LONGER_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != bsize)
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
				throw std::runtime_error("");
			}

			mTotalRcvTime += rcvTime;
			SendMsg(sendAck);
			tResponse.append((char *)recvBuffer.data(), bsize);
		} while (status != 0x40);
	}
	catch (int&)
	{
		if (Reconnect(true) == false) 
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection Error: Connection interrupted and cannot reconnect."));
		}
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection interrupted but reconnected."));
		}
			
		throw std::runtime_error("");
	}
	catch (...)
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection Error: Other error."));
		throw std::runtime_error("");
	}

	GetSystemTime(&et);
	SystemTimeToFileTime(&et, &fet.fT);
	mTotalTime = (unsigned long)((fet.uL.QuadPart - fst.uL.QuadPart) / 10000);
}

bool TSEAccessByEscPosNetwork::DisableTseAccess()
{
	unsigned long rcvTime = 0;

	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd{ 0x10, 0x14, 0x06, 0x01, 0x01, 0x01, 0x03, 0x14, 0x01, 0x06, 0x02, 0x08 };

	mTotalRcvTime = 0;
	mTotalTime = 0;
	SYSTEMTIME st, et;
	union timeunion {
		FILETIME fT;
		ULARGE_INTEGER uL;
	} fst, fet;
	std::stringstream sLogAll;
	GetSystemTime(&st);
	SystemTimeToFileTime(&st, &fst.fT);

	try
	{
		SendMsg(sendCmd);
		while (true)
		{
			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x37) continue;
			mTotalRcvTime += rcvTime;

			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x5C) continue;
			mTotalRcvTime += rcvTime;
			break;
		}
		RecvMsg(recvBuffer, 2, DEFAULT_RCV_TIMEOUT, rcvTime);
		if (recvBuffer.size() != 2)
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed."));
			throw std::runtime_error("");
		}
		mTotalRcvTime += rcvTime;
	}
	catch (int&)
	{
		if (Reconnect(true) == false)
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection Error: Connection interrupted and cannot reconnect."));
		}	
		else
		{
			TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Connection interrupted but reconnected."));
		}

		throw std::runtime_error("");
	}
	catch (...)
	{
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed : Other error."));
		throw std::runtime_error("");
	}

	if (recvBuffer[0] != 0x30)
	{
		return false;
	}

	GetSystemTime(&et);
	SystemTimeToFileTime(&et, &fet.fT);
	mTotalTime = (unsigned long)((fet.uL.QuadPart - fst.uL.QuadPart) / 10000);

	return true;
}

bool TSEAccessByEscPosNetwork::Reconnect(bool reopen)
{
	int retry = 0;
	TSEPRINTLOG(mLog, LIBLOG_LEVEL_TRACE, ("Trying to reconnect.."));
	while (true)
	{
		if (mASocket != NULL) delete mASocket;

		try
		{
			mASocket = new AccessSocket(mDeviceId.c_str(), PRINT_PORT, false, mLog);
		}
		catch (int&)
		{
			mASocket = NULL;
			retry++;
			continue;
		}
		catch (...)
		{
			return false;
		}
		TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("Reconnected! Trying to access again the TSE Device, which may take up to a minute. Please wait."));
		if (reopen)
		{
			Sleep(60000); //Wait for the printer/box to release the TSE device
			unsigned long rcvTime = 0;

			std::vector<BYTE> recvBuffer;
			std::vector<BYTE> tmpBuffer;
			std::vector<BYTE> sendCmdOpen{ 0x10, 0x14, 0x06, 0x01, 0x03, 0x01, 0x03, 0x14, 0x01, 0x06, 0x02, 0x08 };

			while (true) 
			{
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_TRACE, (">>>"));
				try
				{
					RecvMsg(tmpBuffer, 65541, 1000, rcvTime);
				}
				catch (int&)
				{
					//do nothing
				}
				catch (...)
				{
					return false;
				}

				TSEPRINTLOG(mLog, LIBLOG_LEVEL_TRACE, ("<<<"));

				try
				{
					SendMsg(sendCmdOpen);
					RecvMsg(recvBuffer, 4, 1000, rcvTime);
				}
				catch (int&)
				{
					TSEPRINTLOG(mLog, LIBLOG_LEVEL_WARNING, ("This program requires more time to access the TSE Device. Please wait..."));
					continue;
				}
				catch (...)
				{
					return false;
				}

				TSEPRINTLOG(mLog, LIBLOG_LEVEL_TRACE, (">>"));
				while (true)
				{
					try
					{
						RecvMsg(tmpBuffer, 1, 500, rcvTime);
					}
					catch (int&)
					{
						//do nothing
					}
					catch (...)
					{
						return false;
					}
					if (tmpBuffer.size() == 0) break;
				}
				TSEPRINTLOG(mLog, LIBLOG_LEVEL_TRACE, ("<<"));
				
				if ((recvBuffer[0] != 0x37) || (recvBuffer[1] != 0x5C) || (recvBuffer[2] != 0x30))
				{
					continue;
				}
				break;
			}
		}
		break;
	}
	return true;

}

void TSEAccessByEscPosNetwork::SendTo(const std::vector<unsigned char>& tBuffer)
{
	mASocket->EnableForceSend(mDeviceId.c_str());
	mASocket->SocketSend(tBuffer);
}

void TSEAccessByEscPosNetwork::RecvFrom(std::vector<BYTE>& tBuffer, const unsigned long& tMaxSize, const unsigned long& tTimeout)
{
	mASocket->SocketRecv(tBuffer, tMaxSize, tTimeout);
}

//EOF