// Copyright 2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/09  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include <windows.h>
#include "TSELogger.h"
#include "TSEUtil.h"

#include <stdio.h>
#include <stdarg.h>
#include <string>
#include <sstream>
#include <fstream>
#include <iostream>


#define TSEDEBUGBUFFER				200000
#define TSELOGNAME					"EPSONTSELIBC"
#define TSELOGFILEPATH				"\\EPSON\\TSE\\CLIBLogs\\"

TSELogger::TSELogger(std::string& tLogName)
{
	mhMutex = CreateMutex(NULL, FALSE, NULL);
	if (tLogName == "")
	{
		mLogName = TSELOGNAME;
	}
	else
	{
		mLogName = tLogName;
	}
	mLogLevel = LIBLOG_LEVEL_INFO; //LIBLOG_LEVEL_MAX; 
	mLogMethod = TSELOGONFILE;
	mLogFilename = mLogName + ".log";

	if (mLogFilename != "")
	{
		std::string progdatapath = TSEUtil::GetProgramDataPath();

		if (progdatapath == "")
		{
			mLogMethod = mLogMethod & (~((uint8_t)TSELOGONFILE));// remove logging on file
		}
		else
		{
			mLogFilepath = progdatapath + TSELOGFILEPATH + mLogFilename;
			if (TSEUtil::CreateTseDirectory(mLogFilepath) == false)
			{
				mLogMethod = mLogMethod & (~((uint8_t)TSELOGONFILE));// remove logging on file
			}
		}
	}
}

std::string TSELogger::GetDate()
{
	time_t rawtime;
	struct tm timeinfo;
	char buffer[80];

	time(&rawtime);
	localtime_s(&timeinfo, &rawtime);

	strftime(buffer, sizeof(buffer), "%Y-%m-%d %H:%M:%S", &timeinfo);
	return std::string(buffer);
}

std::string TSELogger::GetDateTime()
{
	time_t rawtime;
	struct tm timeinfo;
	char buffer[80];

	time(&rawtime);
	localtime_s(&timeinfo, &rawtime);

	strftime(buffer, sizeof(buffer), "%Y-%m-%d.%H-%M-%S", &timeinfo);
	return std::string(buffer);
}

void TSELogger::SetLogFile(std::string tLogFile)
{
	mLogFilename = tLogFile;

	if (mLogFilename != "") 
	{
		std::string progdatapath = TSEUtil::GetProgramDataPath();

		if (progdatapath == "")
		{
			mLogMethod = mLogMethod & (~((uint8_t)TSELOGONFILE));// remove logging on file
		}
		else
		{
			mLogFilepath = progdatapath + TSELOGFILEPATH + mLogFilename;
			if (TSEUtil::CreateTseDirectory(mLogFilepath) == false)
			{
				mLogMethod = mLogMethod & (~((uint8_t)TSELOGONFILE));// remove logging on file
			}
		}
	}
}

void TSELogger::SetLogLevel(LiblogLevel tLogLevel, uint8_t tLogMethod)
{

	if ((tLogLevel < LIBLOG_LEVEL_DISABLE) || (tLogLevel > LIBLOG_LEVEL_MAX))
	{
		mLogLevel = LIBLOG_LEVEL_DISABLE;
	}
	else
	{
		mLogLevel = tLogLevel;
	}

	mLogMethod = tLogMethod;
	if ((mLogFilename != "") && (tLogMethod & TSELOGONFILE)) 
	{
		std::string progdatapath = TSEUtil::GetProgramDataPath();

		if (progdatapath == "")
		{
			mLogMethod = mLogMethod & (~((uint8_t)TSELOGONFILE));// remove logging on file
		}
		else
		{
			mLogFilepath =  progdatapath + TSELOGFILEPATH + mLogFilename;
			if (TSEUtil::CreateTseDirectory(mLogFilepath) == false)
			{
				mLogMethod = mLogMethod & (~((uint8_t)TSELOGONFILE));// remove logging on file
			}
		}
	}

	return;
}

void TSELogger::ManageLogFile(unsigned long tTrgtLogFileSize)
{
	if (mLogMethod & TSELOGONFILE)
	{
		std::ifstream ifs(mLogFilepath, std::ios::binary);

		if (!ifs.good())
		{
			return;
		}
		const auto begin = ifs.tellg();
		ifs.seekg(0, std::ios::end);
		const auto end = ifs.tellg();
		const auto fsize = (end - begin);
		ifs.close();

		if (fsize > tTrgtLogFileSize)
		{
			std::string dateTime = GetDateTime();
			std::string backUpLog = mLogFilepath + dateTime + ".backup";
			int ret;
			if ((ret = rename(mLogFilepath.c_str(), backUpLog.c_str())) != 0)
			{
				//perror("Error renaming log file");
			}
		}
	}
}

std::string TSELogger::GetLogMessage(const char * tArg, ...)
{
	std::string sbuf = "";
	char buffer[TSEDEBUGBUFFER];
	va_list argv;

	va_start(argv, tArg);
	_vsnprintf_s(buffer, TSEDEBUGBUFFER, tArg, argv);
	va_end(argv);

	return std::string(buffer);
}


std::string TSELogger::MakePrintableString(const std::vector<unsigned char>& tBuffer)
{
	std::string sBuf = "";
	for (unsigned long i = 0; i < tBuffer.size(); i++) {
		switch (tBuffer[i])
		{
			case(0x00): { sBuf += "[0x00]"; break; }
			case(0x01): { sBuf += "[0x01]"; break; }
			case(0x02): { sBuf += "[0x02]"; break; }
			case(0x03): { sBuf += "[0x03]"; break; }
			case(0x04): { sBuf += "[0x04]"; break; }
			case(0x05): { sBuf += "[0x05]"; break; }
			case(0x06): { sBuf += "[0x06]"; break; }
			case(0x07): { sBuf += "[0x07]"; break; }
			case(0x08): { sBuf += "[0x08]"; break; }
			case(0x09): { sBuf += "[0x09]"; break; }
			//	case(0x0a): { sBuf +=  "[0x0A]"; break; }
			case(0x0b): { sBuf += "[0x0B]"; break; }
			case(0x0c): { sBuf += "[0x0C]"; break; }
			//	case(0x0d): { sBuf +=  "[0x0D]"; break; }
			case(0x0e): { sBuf += "[0x0E]"; break; }
			case(0x0f): { sBuf += "[0x0F]"; break; }
			case(0x10): { sBuf += "[0x10]"; break; }
			case(0x11): { sBuf += "[0x11]"; break; }
			case(0x12): { sBuf += "[0x12]"; break; }
			case(0x13): { sBuf += "[0x13]"; break; }
			case(0x14): { sBuf += "[0x14]"; break; }
			case(0x15): { sBuf += "[0x15]"; break; }
			case(0x16): { sBuf += "[0x16]"; break; }
			case(0x17): { sBuf += "[0x17]"; break; }
			case(0x18): { sBuf += "[0x18]"; break; }
			case(0x19): { sBuf += "[0x19]"; break; }
			case(0x1a): { sBuf += "[0x1A]"; break; }
			case(0x1b): { sBuf += "[0x1B]"; break; }
			case(0x1c): { sBuf += "[0x1C]"; break; }
			case(0x1d): { sBuf += "[0x1D]"; break; }
			case(0x1e): { sBuf += "[0x1E]"; break; }
			case(0x1f): { sBuf += "[0x1F]"; break; }
			//	case(0x20): { sBuf +=  "[0x20]"; break; }
			case(0x7F): { sBuf += "[0x7F]"; break; }
			default: {sBuf += tBuffer[i]; break; }
		}
	}

	return sBuf;
}

void TSELogger::PrintLog(unsigned short tDbgLvl, std::string tFileName, unsigned long tLine, std::string tFuncName, std::string tMsg)
{
	std::string sbuf = "";
	std::stringstream ssLine;


	if (tDbgLvl > mLogLevel)
	{
		return;
	}

	ssLine << tLine;

	if (mLogName.length() <= 25)
	{
		sbuf = "[" + mLogName +"]";
	}
	else
	{
		sbuf = "[" + mLogName.substr(mLogName.length() - 25) +"]";
	}

	switch (tDbgLvl)
	{
		case LIBLOG_LEVEL_ERROR:			{sbuf += "[ERROR]"; break; }
		case LIBLOG_LEVEL_WARNING:			{sbuf += "[WARNI]"; break; }
		case LIBLOG_LEVEL_INFO:				{sbuf += "[INFOR]"; break; }
		case LIBLOG_LEVEL_TRACE:			{sbuf += "[TRACE]"; break; }
		case LIBLOG_LEVEL_VERBOSE:			{sbuf += "[VERBO]"; break; }
	}

	sbuf += " " + GetDate();	
	sbuf += " " + tFileName.substr(tFileName.find_last_of("/\\") + 1) + "-" + ssLine.str() + ":" + tFuncName;
	sbuf += ":: " + tMsg;

	if (mLogMethod & TSELOGONDEBUGGER)
	{
		OutputDebugString(sbuf.c_str());
	}

	if (mLogMethod & TSELOGONFILE)
	{
		if (mLogFilename != "")
		{
			WaitForSingleObject(mhMutex, INFINITE);
			std::ofstream ofs(mLogFilepath, std::ios::app);
			if (ofs.good())
			{
				ofs << sbuf << std::endl;
				ofs.close();
			}
			ReleaseMutex(mhMutex);
		}
	}

	if (mLogMethod & TSELOGONSTDOUT)
	{
		HANDLE hConsole = GetStdHandle(STD_OUTPUT_HANDLE);

		SetConsoleTextAttribute(hConsole, FOREGROUND_BLUE | FOREGROUND_GREEN | FOREGROUND_INTENSITY);
		std::cout << sbuf << std::endl;
		SetConsoleTextAttribute(hConsole, FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE);
	}
	return;
}
