// Copyright 2020-2021 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2021/03  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include <string>
#include <sstream>
#include <stdexcept>  
#include "../Include/EpsonTseLibCMulti.h"
#include "../Include/EpsonTseLibStatusCodes.h"
#include "../../Core/TseOperate.h"
#include "../../Core/TseOptions.h"
#include "../../Core/TSELogger.h"
#include "../../Core/TSEDeviceList.h"
#include "EpsonTseLibLocal.h"
#include "TSEMemoryMonitor.h"
#include "EpsonTseErrorDesc.h"
#include "EpsonTseLibVersion.h"

#define EPSON_TSE_ADMIN						"Administrator"
#define PRINT_MEMORY						true
#define ERROR_DESC_SIZE						100

static HANDLE g_hMutex;
TSELogger g_log(std::string("EPSONTSELIBCMULTI"));
static std::list<EPSON_TSE_HANDLE> g_tse_list = {};


BOOL APIENTRY DllMain(HMODULE hModule,
	DWORD  ul_reason_for_call,
	LPVOID lpReserved
)
{
	switch (ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
//	case DLL_THREAD_ATTACH:
		g_hMutex = CreateMutex(NULL, FALSE, NULL);
		g_log.SetLogLevel(LIBLOG_LEVEL_TRACE, TSELOGONFILE);
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_INFO, ("ATTACHED"));
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_INFO, ("DLL Version=%s", EPSONTSELIBC_VERSION_S));
		break;
//	case DLL_THREAD_DETACH:
	case DLL_PROCESS_DETACH:
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_INFO, ("DETACHING..."));
		TSEMemoryDeallocateAll();
		std::list<EPSON_TSE_HANDLE>::const_iterator iterator;

		WaitForSingleObject(g_hMutex, INFINITE);
		for (iterator = g_tse_list.begin(); iterator != g_tse_list.end(); ++iterator)
		{
			TSEOperate * tmp = (TSEOperate *)*iterator;
			delete tmp;
		}
		g_tse_list.clear();
		ReleaseMutex(g_hMutex);
		CloseHandle(g_hMutex);

		TSEPRINTLOG(g_log, LIBLOG_LEVEL_INFO, ("DETACHED"));
		break;
	}
	return TRUE;
}

short int _ExportData(
	TSEOperate * tseOp,
	std::vector<unsigned  char> &exportDataAll,
	bool finalizeExport,
	bool deleteStoredData = false)
{
	std::string result;

	std::string export_result = "";
	std::string export_data = "";
	TSELogger& tLog = tseOp->GetLoggerIntance();

	std::string delete_stored_data = (deleteStoredData) ? "true" : "false";
	exportDataAll.clear();

	if (tseOp == NULL)
	{
		TSEPRINTLOG(tLog, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	do {
		short int status_code;

		export_result = "";
		try 
		{
			result = tseOp->GetExportData(export_data, export_result);
		}
		catch (std::runtime_error&)
		{
			TSEPRINTLOG(tLog, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
			return DEVICE_CONNECTION_ERROR;
		}

		if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
		{
			exportDataAll.clear();
			try
			{
				tseOp->CancelExport();
			}
			catch (std::runtime_error&)
			{
				TSEPRINTLOG(tLog, LIBLOG_LEVEL_WARNING, ("Error in CancelExport"));
			}

			TSEPRINTLOG(tLog, LIBLOG_LEVEL_ERROR, ("%s", result));
			return status_code;
		}

		if (export_result == "EXPORT_INCOMPLETE_SEARCHING")
		{
			continue;
		}

		try
		{
			LDecodeBase64(export_data, exportDataAll);
		}
		catch (std::runtime_error&) 
		{
			exportDataAll.clear();
			try
			{
				tseOp->CancelExport();
			}
			catch (std::runtime_error&)
			{
				TSEPRINTLOG(tLog, LIBLOG_LEVEL_WARNING, ("Error in CancelExport"));
			}
			TSEPRINTLOG(tLog, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
		}
	} while (export_result != "EXPORT_COMPLETE");

	if (finalizeExport)
	{
		try 
		{
			result = tseOp->FinalizeExport(delete_stored_data);
		}
		catch (std::runtime_error&)
		{
			TSEPRINTLOG(tLog, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT"));
			return EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT;
		}
	}

	return EXECUTION_OK;
}

bool IsTseHandleValid(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	bool isExist = false;
	
	WaitForSingleObject(g_hMutex, INFINITE);
	isExist = (std::find(g_tse_list.begin(), g_tse_list.end(), tseHandle) != g_tse_list.end());
	ReleaseMutex(g_hMutex);

	return isExist;
}

/**************** API Functions ****************/

EPSON_TSE_API short int GetTseDeviceList(
	/*in*/  unsigned char		*ipAddress,
	/*in*/  unsigned long int	ipAddressLength,
	/*out*/ unsigned char		**tseDeviceList,
	/*out*/ unsigned long int	*tseDeviceListSize)
{
	TSEPRINTLOG(g_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((tseDeviceList == NULL) ||
		(tseDeviceListSize == NULL))
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string ip_address((const char *)ipAddress, ipAddressLength);
	std::string tse_list = "";
	*tseDeviceListSize = 0;

	try
	{
		TSEGetDeviceList(ip_address, tse_list, g_log);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_CANNOT_GET_TSE_LIST"));
		return OTHER_ERROR_CANNOT_GET_TSE_LIST;
	}

	std::vector<std::string> tokens;
	LStringToVector(tse_list, tokens);
	unsigned long bufsize = 0;

	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{
		std::string tok = *str_iter;
		bufsize += (unsigned long)(tok.length() + 1);
	}

	*tseDeviceListSize = bufsize;

	if (*tseDeviceListSize == 0)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_CANNOT_GET_TSE_LIST"));
		return OTHER_ERROR_CANNOT_GET_TSE_LIST;
	}

	*tseDeviceList = (unsigned char *)TSEEALLOCMEMORY(*tseDeviceListSize);
	if (*tseDeviceList == NULL)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}

	unsigned char *tmp_ptr = *tseDeviceList;
	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{
		std::string tok = *str_iter;

		memcpy(tmp_ptr, tok.c_str(), tok.length());
		tmp_ptr[tok.length()] = '\0';
		tmp_ptr = tmp_ptr + tok.length() + 1;
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(g_log);
#endif

	TSEPRINTLOG(g_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int TSEConnect(
	/*in*/  unsigned short int	method,
	/*in*/  unsigned char		*portId,
	/*in*/  unsigned long int	portIdLength,
	/*in*/  unsigned char		*tseId,
	/*in*/  unsigned long int	tseIdLength,
	/*out*/ EPSON_TSE_HANDLE *	tseHandle)
{
	if (tseHandle == NULL)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	if ((portId == NULL) ||
		(tseId == NULL))
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string port_id((const char *)portId, portIdLength);
	std::string tse_id((const char *)tseId, tseIdLength);

	TSEOptions tse_opt;
	tse_opt.mIdentifier = port_id;
	tse_opt.mMethod = method;

	if (tse_id != "") 
	{
		tse_opt.mTseHandler = tse_id;
	}
	else
	{
		tse_opt.mTseHandler = "local_TSE";
	}

	std::list<EPSON_TSE_HANDLE>::const_iterator iterator;
	EPSON_TSE_HANDLE tmpHandle = 0;

	WaitForSingleObject(g_hMutex, INFINITE);
	for (iterator = g_tse_list.begin(); iterator != g_tse_list.end(); ++iterator) 
	{
		TSEOperate * tmp = (TSEOperate *) *iterator;
		if ( (tmp->GetOperateHostId() == std::string((char *)portId)) && 
			 (tmp->GetOperateTseId() == std::string((char *)tseId)) )
		{
			ReleaseMutex(g_hMutex);

			TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR - CONNECTION ALREADY EXISTS i=%s, t=%s", portId, tseId));
			return DEVICE_CONNECTION_ERROR;
		}
	}
	ReleaseMutex(g_hMutex);

	TSEOperate * tse_handle = NULL;
	
	try
	{
		tse_handle = new TSEOperate(tse_opt);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_INFO, ("DLL Version=%s", EPSONTSELIBC_VERSION_S));
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("m=%d i=%s t=%s h=0x%x", method, port_id.c_str(), tse_id.c_str(), tse_handle));
	TSEPRINTLOG(g_log, LIBLOG_LEVEL_TRACE, ("m=%d i=%s t=%s h=0x%x", method, port_id.c_str(), tse_id.c_str(), tse_handle));

	WaitForSingleObject(g_hMutex, INFINITE);
	g_tse_list.push_back((EPSON_TSE_HANDLE)tse_handle);
	ReleaseMutex(g_hMutex);

	*tseHandle = (EPSON_TSE_HANDLE)tse_handle;
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int TSEDisconnect(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("DISCONNECT"));
	if (tse_handle != NULL)
	{
		WaitForSingleObject(g_hMutex, INFINITE);
		g_tse_list.remove((EPSON_TSE_HANDLE)tse_handle);
		ReleaseMutex(g_hMutex);

		try
		{	
			delete(tse_handle);
		}
		catch (std::runtime_error&)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_WARNING, ("Cannot delete g_tse"));
			TSEPRINTLOG(g_log, LIBLOG_LEVEL_WARNING, ("Cannot delete g_tse"));
		}
	}
	TSEPRINTLOG(g_log, LIBLOG_LEVEL_TRACE, ("DISCONNECT h=0x%x", tseHandle));
	return EXECUTION_OK;
}


EPSON_TSE_API short int TSEOpenDevice(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		tse_handle->TSEOpenDevice();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int TSECloseDevice(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try 
	{
		tse_handle->TSECloseDevice();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int Setup(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*adminPin,
	/*in*/  unsigned long int	adminPinLength,
	/*in*/  unsigned char		*clientIdPin,
	/*in*/  unsigned long int	clientIdPinLength,
	/*in*/  unsigned char		*puk,
	/*in*/  unsigned long int	pukLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((adminPin == NULL) ||
		(clientIdPin == NULL) ||
		(puk == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string admin_pin((const char *)adminPin, adminPinLength);
	std::string time_admin_pin((const char *)clientIdPin, clientIdPinLength);
	std::string puk_s((const char *)puk, pukLength);

	try
	{
		result = tse_handle->Setup(puk_s, admin_pin, time_admin_pin);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int RunTSESelfTest(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->RunTSESelfTest();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int FactoryReset(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->FactoryReset();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	try
	{
		result = tse_handle->RunTSESelfTest();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if (((status_code = LGetStatusCode(result)) != EXECUTION_OK) && (status_code != TSE1_ERROR_CLIENT_NOT_REGISTERED))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetRawStorageInfo(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned char		**rawTseInfo,
	/*out*/ unsigned long int	*rawTseInfoLength
	)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((rawTseInfo == NULL) ||
		(rawTseInfoLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string tse_info;

	try
	{
		result = tse_handle->GetStorageInfo(tse_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	*rawTseInfoLength = (unsigned long)tse_info.length();
	*rawTseInfo = (unsigned char *)TSEEALLOCMEMORY(*rawTseInfoLength + NULL_CHAR);
	if (*rawTseInfo == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LStringToCharArray(tse_info, *rawTseInfo, (*rawTseInfoLength + NULL_CHAR));

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetStorageInfo(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ EPSON_TSE_INFO		*tseInfo 
	)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));

	if (tseInfo == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string tse_info;

	try
	{
		result = tse_handle->GetStorageInfo(tse_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	memset(tseInfo->vendorType, '\0', 100 + NULLCHAR);
	memset(tseInfo->tseInitializationState, '\0', 100 + NULLCHAR);
	memset(tseInfo->tseDescription, '\0', 128 + NULLCHAR);
	memset(tseInfo->signatureAlgorithm, '\0', 100 + NULLCHAR);
	memset(tseInfo->cdcId, '\0', 18 + NULLCHAR);
	memset(tseInfo->cdcHash, '\0', 64 + NULLCHAR);
	memset(tseInfo->certificateExpirationDateStr, '\0', 30 + NULLCHAR);
	memset(tseInfo->lastExportExecutedDateStr, '\0', 30 + NULLCHAR);

	std::string created_signatures = LJsonGetNonStringValue(tse_info, "createdSignatures");
	tseInfo->createdSignatures = LStringToLongInt(created_signatures);

	std::string hardward_version = LJsonGetNonStringValue(tse_info, "hardwareVersion");
	tseInfo->hardwareVersion = LStringToLongInt(hardward_version);

	std::string has_passed_self_test = LJsonGetNonStringValue(tse_info, "hasPassedSelfTest");
	tseInfo->hasPassedSelfTest = LStringToBool(has_passed_self_test);

	std::string has_valid_time = LJsonGetNonStringValue(tse_info, "hasValidTime");
	tseInfo->hasValidTime = LStringToBool(has_valid_time);

	std::string is_transaction_in_progress = LJsonGetNonStringValue(tse_info, "isTransactionInProgress");
	tseInfo->isTransactionInProgress = LStringToBool(is_transaction_in_progress);

	std::string is_export_enabled_if_csp_test_fails = LJsonGetNonStringValue(tse_info, "isExportEnabledIfCspTestFails");
	tseInfo->isExportEnabledIfCspTestFails = LStringToBool(is_export_enabled_if_csp_test_fails);

	std::string is_tse_unlocked = LJsonGetNonStringValue(tse_info, "isTSEUnlocked");
	tseInfo->isTSEUnlocked = LStringToBool(is_tse_unlocked);

	std::string max_registered_clients = LJsonGetNonStringValue(tse_info, "maxRegisteredClients");
	tseInfo->maxRegisteredClients = LStringToLongInt(max_registered_clients);

	std::string max_signatures = LJsonGetNonStringValue(tse_info, "maxSignatures");
	tseInfo->maxSignatures = LStringToLongInt(max_signatures);

	std::string max_started_transactions = LJsonGetNonStringValue(tse_info, "maxStartedTransactions");
	tseInfo->maxStartedTransactions = LStringToLongInt(max_started_transactions);

	std::string maxUpdate_delay = LJsonGetNonStringValue(tse_info, "maxUpdateDelay");
	tseInfo->maxUpdateDelay = LStringToLongInt(maxUpdate_delay);

	std::string registered_clients = LJsonGetNonStringValue(tse_info, "registeredClients");
	tseInfo->registeredClients = LStringToLongInt(registered_clients);

	std::string remaining_signatures = LJsonGetNonStringValue(tse_info, "remainingSignatures");
	tseInfo->remainingSignatures = LStringToLongInt(remaining_signatures);

	std::string software_version = LJsonGetNonStringValue(tse_info, "softwareVersion");
	tseInfo->softwareVersion = LStringToLongInt(software_version);

	std::string started_transactions = LJsonGetNonStringValue(tse_info, "startedTransactions");
	tseInfo->startedTransactions = LStringToLongInt(started_transactions);

	std::string tar_export_size = LJsonGetNonStringValue(tse_info, "tarExportSize");
	tseInfo->tarExportSize = LStringToLongInt(tar_export_size);

	std::string time_until_next_self_test = LJsonGetNonStringValue(tse_info, "timeUntilNextSelfTest");
	tseInfo->timeUntilNextSelfTest = LStringToLongInt(time_until_next_self_test);

	std::string tse_capacity = LJsonGetNonStringValue(tse_info, "tseCapacity");
	tseInfo->tseCapacity = LStringToLongInt(tse_capacity);

	std::string tse_current_size = LJsonGetNonStringValue(tse_info, "tseCurrentSize");
	tseInfo->tseCurrentSize = LStringToLongInt(tse_current_size);

	std::string certificate_expiration_date = LJsonGetValue(tse_info, "certificateExpirationDate");
	LStringToUtcDateTime(certificate_expiration_date, tseInfo->certificateExpirationDate);
	LStringToCharArray(certificate_expiration_date, tseInfo->certificateExpirationDateStr, sizeof(tseInfo->certificateExpirationDateStr));

	std::string last_export_executed_date = LJsonGetValue(tse_info, "lastExportExecutedDate");
	LStringToUtcDateTime(last_export_executed_date, tseInfo->lastExportExecutedDate);
	LStringToCharArray(last_export_executed_date, tseInfo->lastExportExecutedDateStr, sizeof(tseInfo->lastExportExecutedDateStr));

	std::string signature_algorithm = LJsonGetValue(tse_info, "signatureAlgorithm");
	LStringToCharArray(signature_algorithm, tseInfo->signatureAlgorithm, sizeof(tseInfo->signatureAlgorithm));

	std::string tse_description = LJsonGetValue(tse_info, "tseDescription");
	LStringToCharArray(tse_description, tseInfo->tseDescription, sizeof(tseInfo->tseDescription));

	std::string tse_initialization_state = LJsonGetValue(tse_info, "tseInitializationState");
	LStringToCharArray(tse_initialization_state, tseInfo->tseInitializationState, sizeof(tseInfo->tseInitializationState));

	std::string vendor_type = LJsonGetValue(tse_info, "vendorType");
	LStringToCharArray(vendor_type, tseInfo->vendorType, sizeof(tseInfo->vendorType));

	std::string cdc_id = LJsonGetValue(tse_info, "cdcId");
	LStringToCharArray(cdc_id, tseInfo->cdcId, sizeof(tseInfo->cdcId));

	std::string cdc_hash = LJsonGetValue(tse_info, "cdcHash");
	LStringToCharArray(cdc_hash, tseInfo->cdcHash, sizeof(tseInfo->cdcHash));

	std::string serial_number = LJsonGetValue(tse_info, "serialNumber");
	std::string tse_public_key = LJsonGetValue(tse_info, "tsePublicKey");

	tseInfo->rawTseInfoLength = (unsigned long)tse_info.length();

	std::vector<unsigned char> decoded_stream_ser;
	std::vector<unsigned char> decoded_stream_key;
	try
	{
		LDecodeBase64(serial_number, decoded_stream_ser);
		LDecodeBase64(tse_public_key, decoded_stream_key);
	}
	catch (std::runtime_error&) {
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
	}

	tseInfo->serialNumberLength = (unsigned long)decoded_stream_ser.size();
	tseInfo->tsePublicKeyLength = (unsigned long)decoded_stream_key.size();

	tseInfo->serialNumber = (unsigned char *)TSEEALLOCMEMORY(tseInfo->serialNumberLength);
	if (tseInfo->serialNumber == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream_ser, tseInfo->serialNumber, tseInfo->serialNumberLength);

	tseInfo->tsePublicKey = (unsigned char *)TSEEALLOCMEMORY(tseInfo->tsePublicKeyLength);
	if (tseInfo->tsePublicKey == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream_key, tseInfo->tsePublicKey, tseInfo->tsePublicKeyLength);

	tseInfo->rawTseInfo = (unsigned char *)TSEEALLOCMEMORY(tseInfo->rawTseInfoLength + NULL_CHAR);
	if (tseInfo->rawTseInfo == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LStringToCharArray(tse_info, tseInfo->rawTseInfo, (tseInfo->rawTseInfoLength + NULL_CHAR));

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int GetRawStorageSmartInfo(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned char		**smartInfo,
	/*out*/ unsigned long int	*smartInfoLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((smartInfo == NULL) ||
		(smartInfoLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string smart_info;

	try
	{
		result = tse_handle->GetStorageSmartInfo(smart_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	*smartInfoLength = (unsigned long)smart_info.length();
	*smartInfo = (unsigned char *)TSEEALLOCMEMORY(*smartInfoLength + NULL_CHAR);
	if (*smartInfo == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LStringToCharArray(smart_info, *smartInfo, (*smartInfoLength + NULL_CHAR));

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetStorageSmartInfo(
	/*in*/  EPSON_TSE_HANDLE		tseHandle,
	/*out*/ EPSON_TSE_SMART_INFO	*smartInfo)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (smartInfo == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string smart_info;

	try
	{
		result = tse_handle->GetStorageSmartInfo(smart_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	memset(smartInfo->dataIntegrity_healthStatus, '\0', 64 + NULLCHAR);
	memset(smartInfo->eraseLifetimeStatus_healthStatus, '\0', 64 + NULLCHAR);
	memset(smartInfo->spareBlockStatus_healthStatus, '\0', 64 + NULLCHAR);
	memset(smartInfo->tseHealth, '\0', 64 + NULLCHAR);

	std::string data_integrity = LJsonGetObj(smart_info, "dataIntegrity");
	std::string erase_lifetime_status = LJsonGetObj(smart_info, "eraseLifetimeStatus");
	std::string spare_block_status = LJsonGetObj(smart_info, "spareBlockStatus");

	std::string is_replacement_needed = LJsonGetNonStringValue(smart_info, "isReplacementNeeded");
	smartInfo->isReplacementNeeded = LStringToBool(is_replacement_needed);

	std::string remaining_data_retention = LJsonGetNonStringValue(smart_info, "remainingTenYearsDataRetention");
	smartInfo->remainingTenYearsDataRetention = LStringToLongInt(remaining_data_retention);

	std::string tse_health = LJsonGetValue(smart_info, "tseHealth");
	LStringToCharArray(tse_health, smartInfo->tseHealth, sizeof(smartInfo->tseHealth));

	std::string dI_healthStatus = LJsonGetValue(data_integrity, "healthStatus");
	LStringToCharArray(dI_healthStatus, smartInfo->dataIntegrity_healthStatus, sizeof(smartInfo->dataIntegrity_healthStatus));

	std::string dI_uncorrectableECCErrors = LJsonGetNonStringValue(data_integrity, "uncorrectableECCErrors");
	smartInfo->dataIntegrity_uncorrectableECCErrors = LStringToLongInt(dI_uncorrectableECCErrors);


	std::string eL_healthStatus = LJsonGetValue(erase_lifetime_status, "healthStatus");
	LStringToCharArray(eL_healthStatus, smartInfo->eraseLifetimeStatus_healthStatus, sizeof(smartInfo->eraseLifetimeStatus_healthStatus));

	std::string eL_remainingEraseCounts = LJsonGetNonStringValue(erase_lifetime_status, "remainingEraseCounts");
	smartInfo->eraseLifetimeStatus_remainingEraseCounts = LStringToLongInt(eL_remainingEraseCounts);

	std::string sB_healthStatus = LJsonGetValue(spare_block_status, "healthStatus");
	LStringToCharArray(sB_healthStatus, smartInfo->spareBlockStatus_healthStatus, sizeof(smartInfo->spareBlockStatus_healthStatus));

	std::string sB_remainingSpareBlocks = LJsonGetNonStringValue(spare_block_status, "remainingSpareBlocks");
	smartInfo->spareBlockStatus_remainingSpareBlocks = LStringToLongInt(sB_remainingSpareBlocks);

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetStartedTransactionList(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*out*/ unsigned long int	**startedTransactionList,
	/*out*/ unsigned long int	*startedTransactionListSize)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((clientId == NULL) ||
		(startedTransactionList == NULL) ||
		(startedTransactionListSize == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::string trx_list = "";

	try 
	{
		result = tse_handle->GetStartedTransactionList(client_id, trx_list);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::vector<unsigned long int> tokens;
	LStringToVector(trx_list, tokens);

	*startedTransactionListSize = (unsigned long) tokens.size();
	*startedTransactionList = (unsigned long int *)TSEEALLOCMEMORY(*startedTransactionListSize * sizeof(unsigned long int));
	if (*startedTransactionList == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}

	int i = 0;
	for (std::vector<unsigned long int>::iterator uint_iter = tokens.begin(); uint_iter != tokens.end(); ++uint_iter)
	{
		(*startedTransactionList)[i] = *uint_iter;
		i++;
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetLastTransactionResponse(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*out*/ unsigned long int	*transactionNumber,
	/*out*/ unsigned long int	*logTime,
	/*out*/ unsigned long int	*signatureCounter,
	/*out*/ unsigned char		**signatureValue,
	/*out*/ unsigned long int	*signatureValueLength,
	/*out*/ unsigned char		**serialNumber,
	/*out*/ unsigned long int	*serialNumberLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((clientId == NULL) ||
		(transactionNumber == NULL) ||
		(logTime == NULL) ||
		(serialNumber == NULL) ||
		(serialNumberLength == NULL) ||
		(signatureCounter == NULL) ||
		(signatureValue == NULL) ||
		(signatureValueLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::string trx_no;
	std::string log_time;
	std::string signature_ctr;
	std::string signature_val;
	std::string serial_no;

	try
	{
		result = tse_handle->GetLastTransactionResponse(
			client_id,
			trx_no,
			log_time,
			signature_ctr,
			signature_val,
			serial_no);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	*transactionNumber = LStringToLongInt(trx_no);
	*signatureCounter = LStringToLongInt(signature_ctr);
	LStringToUtcDateTime(log_time, *logTime);

	std::vector<unsigned char> decoded_stream_ser;
	std::vector<unsigned char> decoded_stream_sig;
	try
	{
		LDecodeBase64(serial_no, decoded_stream_ser);
		LDecodeBase64(signature_val, decoded_stream_sig);
	}
	catch (std::runtime_error&) {
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
	}

	*serialNumberLength = (unsigned long)decoded_stream_ser.size();
	*signatureValueLength = (unsigned long)decoded_stream_sig.size();

	*signatureValue = (unsigned char *)TSEEALLOCMEMORY(*signatureValueLength);
	if (*signatureValue == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream_sig, *signatureValue, *signatureValueLength);

	*serialNumber = (unsigned char *)TSEEALLOCMEMORY(*serialNumberLength);
	if (*serialNumber == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream_ser, *serialNumber, *serialNumberLength);

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetMaxNumberOfClients(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned long int	*maxNumberClients)
{
	short int status_code;
	std::string result;
	std::string tse_info;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (maxNumberClients == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	try
	{
		result = tse_handle->GetStorageInfo(tse_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::string max_registered_clients = LJsonGetNonStringValue(tse_info, "maxRegisteredClients");
	*maxNumberClients = LStringToLongInt(max_registered_clients);

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetCurrentNumberOfClients(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned long int	*currentNumberClients)
{
	short int status_code;
	std::string result;
	std::string tse_info;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (currentNumberClients == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	try
	{
		result = tse_handle->GetStorageInfo(tse_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::string registered_clients = LJsonGetNonStringValue(tse_info, "registeredClients");
	*currentNumberClients = LStringToLongInt(registered_clients);

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetMaxNumberOfTransactions(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned long int	*maxNumberTransactions)
{
	short int status_code;
	std::string result;
	std::string tse_info;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (maxNumberTransactions == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	try
	{
		result = tse_handle->GetStorageInfo(tse_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::string max_started_transactions = LJsonGetNonStringValue(tse_info, "maxStartedTransactions");
	*maxNumberTransactions = LStringToLongInt(max_started_transactions);

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;

}

EPSON_TSE_API short int GetCurrentNumberOfTransactions(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned long int	*currentNumberTransactions)
{
	short int status_code;
	std::string result;
	std::string tse_info;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (currentNumberTransactions == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	try
	{
		result = tse_handle->GetStorageInfo(tse_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::string started_transactions = LJsonGetNonStringValue(tse_info, "startedTransactions");
	*currentNumberTransactions = LStringToLongInt(started_transactions);

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ExportCertificate(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned char		**certificate,
	/*out*/ unsigned long int	*certificateLength)
{
	short int status_code;
	std::string result;
	std::string certificate_s;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((certificate == NULL) ||
		(certificateLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	try
	{
		result = tse_handle->GetLogMessageCertificate(certificate_s);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::vector<unsigned char> decoded_stream;
	try
	{
		LDecodeBase64(certificate_s, decoded_stream);
	}
	catch (std::runtime_error&) {
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
	}
	*certificateLength = (unsigned long) decoded_stream.size();
	*certificate = (unsigned char *)TSEEALLOCMEMORY(*certificateLength);
	if (*certificate == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream, *certificate, *certificateLength);

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ExportSerialNumber(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned char		**serialNumber,
	/*out*/ unsigned long int	*serialNumberLength)
{
	short int status_code;
	std::string result;
	std::string tse_info;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((serialNumber == NULL) ||
		(serialNumberLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	try
	{
		result = tse_handle->GetStorageInfo(tse_info);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::string serial_number = LJsonGetValue(tse_info, "serialNumber");
	std::vector<unsigned char> decoded_stream;
	try
	{
		LDecodeBase64(serial_number, decoded_stream);
	}
	catch (std::runtime_error&) {
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
	}
	*serialNumberLength = (unsigned long) decoded_stream.size();
	*serialNumber = (unsigned char *)TSEEALLOCMEMORY(*serialNumberLength);
	if (*serialNumber == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream, *serialNumber, *serialNumberLength);

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int AuthenticateUser(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*userId,
	/*in*/  unsigned long int	userIdLength,
	/*in*/  unsigned char		*pin,
	/*in*/  unsigned long int	pinLength,
	/*in*/  unsigned char		*secretKey,
	/*in*/  unsigned long int	secretKeyLength,
	/*out*/ short int			*remainingRetries)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((userId == NULL) ||
		(pin == NULL) ||
		(secretKey == NULL) ||
		(remainingRetries == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string user_id((const char *)userId, userIdLength);
	std::string pin_s((const char *)pin, pinLength);
	std::string secret_key((const char *)secretKey, secretKeyLength);
	std::string remaining_retries;
	std::string challenge;
	std::string hash = "";
	std::string cha_sctkey_s = "";
	*remainingRetries = -1;

	try
	{
		result = tse_handle->GetChallenge(user_id, challenge);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}
	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	cha_sctkey_s = challenge + secret_key;
	std::vector<unsigned char> cha_sctkey_v(cha_sctkey_s.begin(), cha_sctkey_s.end());

	try
	{
		hash = LEncodeBase64Sha256(cha_sctkey_v);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_ENCODING_FAILED"));
		return OTHER_ERROR_ENCODING_FAILED;
	}

	try
	{
		if (user_id == EPSON_TSE_ADMIN)
		{
			result = tse_handle->AuthenticateUserForAdmin(user_id, pin_s, hash, remaining_retries);
		}
		else
		{
			result = tse_handle->AuthenticateUserForTimeAdmin(user_id, pin_s, hash, remaining_retries);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}


	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		if (status_code == TSE1_ERROR_AUTHENTICATION_FAILED)
		{
			*remainingRetries = LStringToShortInt(remaining_retries);
		}
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int LogOut(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*userId,
	/*in*/  unsigned long int	userIdLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (userId == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string user_id((const char *)userId, userIdLength);

	try
	{
		if (user_id == EPSON_TSE_ADMIN)
		{
			result = tse_handle->LogOutForAdmin();
		}
		else
		{
			result = tse_handle->LogOutForTimeAdmin(user_id);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}
	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ChangePin(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*userId,
	/*in*/  unsigned long int	userIdLength,
	/*in*/  unsigned char		*oldPin,
	/*in*/  unsigned long int	oldPinLength,
	/*in*/  unsigned char		*newPin,
	/*in*/  unsigned long int	newPinLength,
	/*out*/ short int			*remainingRetries)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((userId == NULL) ||
		(oldPin == NULL) ||
		(newPin == NULL) ||
		(remainingRetries == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string user_id((const char *)userId, userIdLength);
	std::string old_pin((const char *)oldPin, oldPinLength);
	std::string new_pin((const char *)newPin, newPinLength);
	std::string remaining_retries;

	*remainingRetries = -1;

	try
	{
		if (user_id == EPSON_TSE_ADMIN)
		{
			result = tse_handle->ChangePinForAdmin(EPSON_TSE_ADMIN, old_pin, new_pin, remaining_retries);
		}
		else
		{
			result = tse_handle->ChangePinForTimeAdmin(user_id, old_pin, new_pin, remaining_retries);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		if (status_code == TSE1_ERROR_AUTHENTICATION_FAILED)
		{
			*remainingRetries = LStringToShortInt(remaining_retries);
		}
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int UpdateTime(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*userId,
	/*in*/  unsigned long int	userIdLength,
	/*in*/  unsigned long int	newDateTime,
	/*in*/  bool				updateForFirstTime)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((userId == NULL) ||
		(newDateTime == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string user_id((const char *)userId, userIdLength);
	std::string new_date_time;
	std::string use_time_sync = "false";

	new_date_time = LUtcDateTimeToString(newDateTime);

	try
	{
		if (updateForFirstTime)
		{
			result = tse_handle->UpdateTimeForFirst(user_id, new_date_time, use_time_sync);
		}
		else
		{
			result = tse_handle->UpdateTime(user_id, new_date_time, use_time_sync);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

#if 0 //Time Sync Not supported yet
EPSON_TSE_API short int UpdateTimeWithTimeSync(
	/*in*/  unsigned char		*userId,
	/*in*/  unsigned long int	userIdLength,
	/*in*/  bool				updateForFirstTime)
{
	short int status_code;
	std::string result;

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (tse == NULL)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	if (userId == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string user_id((const char *)userId, userIdLength);
	std::string new_date_time;
	std::string use_time_sync = "true";

	try
	{
		if (updateForFirstTime)
		{
			result = g_tse->UpdateTimeForFirst(user_id, "", use_time_sync);
		}
		else
		{
			result = g_tse->UpdateTime(user_id, "", use_time_sync);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}
#endif

EPSON_TSE_API short int DisableSecureElement(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try 
	{
		result = tse_handle->DisableSecureElement();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int RegisterTSEToHost(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->SetUpForPrinter();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int RegisterSecretKey(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*secretKey,
	/*in*/  unsigned long int	secretKeyLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (secretKey == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string secret_key((const char *)secretKey, secretKeyLength);

	try
	{
		result = tse_handle->RegisterSecretKey(secret_key);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int RegisterClient(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*newClientId,
	/*in*/  unsigned long int	newClientIdLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (newClientId == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string new_client_id((const char *)newClientId, newClientIdLength);

	try
	{
		result = tse_handle->RegisterClient(new_client_id);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int DeregisterClient(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (clientId == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);

	try
	{
		result = tse_handle->DeregisterClient(client_id);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int LockTse(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->LockTSE();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int UnlockTse(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->UnlockTSE();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int EnableExportIfCspTestFails(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->EnableExportIfCspTestFails();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int DisableExportIfCspTestFails(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->DisableExportIfCspTestFails();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int SetTimeOutInterval(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned long int	timeoutIntervalForAdmin,
	/*in*/  unsigned long int	timeoutIntervalForTimeAdmin,
	/*in*/  unsigned long int	timeoutIntervalForExport)
{
	short int status_code;
	std::string result;
	std::string timeout_interval_admin = "900";
	std::string timeout_interval_timeadmin = "28800";
	std::string timeout_interval_export = "100";

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if (timeoutIntervalForAdmin != 0)
	{
		timeout_interval_admin = LIntegerToString(timeoutIntervalForAdmin);
	}

	if (timeoutIntervalForTimeAdmin != 0)
	{
		timeout_interval_timeadmin = LIntegerToString(timeoutIntervalForTimeAdmin);
	}

	if (timeoutIntervalForExport != 0)
	{
		timeout_interval_export = LIntegerToString(timeoutIntervalForExport);
	}

	try
	{
		result = tse_handle->SetTimeOutInterval(
			timeout_interval_admin,
			timeout_interval_timeadmin,
			timeout_interval_export);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int GetTimeOutInterval(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned long int	*timeoutIntervalForAdmin,
	/*out*/ unsigned long int	*timeoutIntervalForTimeAdmin,
	/*out*/ unsigned long int	*timeoutIntervalForExport)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((timeoutIntervalForAdmin == NULL) ||
		(timeoutIntervalForTimeAdmin == NULL) ||
		(timeoutIntervalForExport == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string timeout_interval_admin;
	std::string timeout_interval_timeadmin;
	std::string timeout_interval_export;

	try
	{
		result = tse_handle->GetTimeOutInterval(
			timeout_interval_admin,
			timeout_interval_timeadmin,
			timeout_interval_export);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}
	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	*timeoutIntervalForAdmin = LStringToLongInt(timeout_interval_admin);
	*timeoutIntervalForTimeAdmin = LStringToLongInt(timeout_interval_timeadmin);
	*timeoutIntervalForExport = LStringToLongInt(timeout_interval_export);

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetAuthenticatedUserList(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*userRole,
	/*in*/  unsigned long int	userRoleLength,
	/*out*/ unsigned char		**AuthenticatedUserList,
	/*out*/ unsigned long int	*AuthenticatedUserListLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((userRole == NULL) ||
		(AuthenticatedUserList == NULL) ||
		(AuthenticatedUserListLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string user_role((const char *)userRole, userRoleLength);
	std::string user_list = "";

	try
	{
		result = tse_handle->GetAuthenticatedUserList(user_role, user_list);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::vector<std::string> tokens;
	LStringToVector(user_list, tokens);
	unsigned long bufsize = 0;

	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{
		std::string tok = *str_iter;
		size_t index = 0;
		while (true)
		{
			index = tok.find("\\/", index);
			if (index == std::string::npos)
				break;
			tok.replace(index, 2, "/");
			index += 1;
		}
		bufsize += (unsigned long) (tok.length() + 1);
	}

	*AuthenticatedUserListLength = bufsize;
	if (*AuthenticatedUserListLength == 0)
	{
		return status_code;
	}

	*AuthenticatedUserList = (unsigned char *)TSEEALLOCMEMORY(*AuthenticatedUserListLength);
	if (*AuthenticatedUserList == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}

	unsigned char *tmp_ptr = *AuthenticatedUserList;
	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{
		std::string tok = *str_iter;
		size_t index = 0;
		while (true)
		{
			index = tok.find("\\/", index);
			if (index == std::string::npos)
				break;
			tok.replace(index, 2, "/");
			index += 1;
		}

		memcpy(tmp_ptr, tok.c_str(), tok.length());
		tmp_ptr[tok.length()] = '\0';
		tmp_ptr = tmp_ptr + tok.length() + 1;
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int GetRegisteredClientList(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned char		**registeredClientList,
	/*out*/ unsigned long int	*registeredClientListLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((registeredClientList == NULL) ||
		(registeredClientListLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string clientList = "";

	try
	{
		result = tse_handle->GetRegisteredClientList(clientList);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	std::vector<std::string> tokens;
	LStringToVector(clientList, tokens);
	unsigned long bufsize = 0;

	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{
		std::string tok = *str_iter;

		size_t index = 0;
		while (true)
		{
			index = tok.find("\\/", index);
			if (index == std::string::npos) 
				break;
			tok.replace(index, 2, "/");
			index += 1;
		}

		bufsize += (unsigned long)(tok.length() + 1);
	}

	*registeredClientListLength = bufsize;

	if (*registeredClientListLength == 0)
	{
		return status_code;
	}

	*registeredClientList = (unsigned char *)TSEEALLOCMEMORY(*registeredClientListLength);
	if (*registeredClientList == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}

	unsigned char *tmp_ptr = *registeredClientList;
	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{
		std::string tok = *str_iter;
		size_t index = 0;
		while (true)
		{
			index = tok.find("\\/", index);
			if (index == std::string::npos)
				break;
			tok.replace(index, 2, "/");
			index += 1;
		}
		memcpy(tmp_ptr, tok.c_str(), tok.length());
		tmp_ptr[tok.length()] = '\0';
		tmp_ptr = tmp_ptr + tok.length() + 1;
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int StartTransaction(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*in*/  unsigned char		*processData,
	/*in*/  unsigned long int	processDataLength,
	/*in*/  unsigned char		*processType,
	/*in*/  unsigned long int	processTypeLength,
	/*in*/  unsigned char		*additionalData,
	/*in*/  unsigned long int	additionalDataLength,
	/*out*/ unsigned long int	*transactionNumber,
	/*out*/ unsigned long int	*logTime,
	/*out*/ unsigned char		**serialNumber,
	/*out*/ unsigned long int	*serialNumberLength,
	/*out*/ unsigned long int	*signatureCounter,
	/*out*/ unsigned char		**signatureValue,
	/*out*/ unsigned long int	*signatureValueLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((clientId == NULL) ||
		(processData == NULL) ||
		(processType == NULL) ||
		(additionalData == NULL) ||
		(transactionNumber == NULL) ||
		(logTime == NULL) ||
		(serialNumber == NULL) ||
		(serialNumberLength == NULL) ||
		(signatureCounter == NULL) ||
		(signatureValue == NULL) ||
		(signatureValueLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::vector<unsigned char> process_data(processData, processData + processDataLength);
	std::string process_type((const char *)processType, processTypeLength);
	std::vector<unsigned char> additnl_data(additionalData, additionalData + additionalDataLength);
	std::string trx_no;
	std::string log_time;
	std::string serial_no;
	std::string signature_ctr;
	std::string signature_val;

	std::string process_data_enc = "";
	std::string addtnl_data_enc = "";

	try
	{
		if (process_data.size() != 0) {
			process_data_enc = LEncodeBase64(process_data);
		}
		if (additnl_data.size() != 0) {
			addtnl_data_enc = LEncodeBase64(additnl_data);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_ENCODING_FAILED"));
		return OTHER_ERROR_ENCODING_FAILED;
	}

	try 
	{
		result = tse_handle->StartTransaction(client_id,
			process_data_enc,
			process_type,
			addtnl_data_enc,
			trx_no,
			log_time,
			serial_no,
			signature_ctr,
			signature_val);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	*transactionNumber = LStringToLongInt(trx_no);
	*signatureCounter = LStringToLongInt(signature_ctr);
	LStringToUtcDateTime(log_time, *logTime);

	std::vector<unsigned char> decoded_stream_ser;
	std::vector<unsigned char> decoded_stream_sig;
	try
	{
		LDecodeBase64(serial_no, decoded_stream_ser);
		LDecodeBase64(signature_val, decoded_stream_sig);
	}
	catch (std::runtime_error&) {
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
	}

	*serialNumberLength = (unsigned long) decoded_stream_ser.size();
	*signatureValueLength = (unsigned long) decoded_stream_sig.size();
	
	*signatureValue = (unsigned char *)TSEEALLOCMEMORY(*signatureValueLength);
	if (*signatureValue == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream_sig, *signatureValue, *signatureValueLength);

	*serialNumber = (unsigned char *)TSEEALLOCMEMORY(*serialNumberLength);
	if (*serialNumber == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream_ser, *serialNumber, *serialNumberLength);

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int UpdateTransaction(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*in*/  unsigned long int	transactionNumber,
	/*in*/  unsigned char		*processData,
	/*in*/  unsigned long int	processDataLength,
	/*in*/  unsigned char		*processType,
	/*in*/  unsigned long int	processTypeLength,
	/*out*/ unsigned long int	*logTime,
	/*out*/ unsigned char		**signatureValue,
	/*out*/ unsigned long int	*signatureValueLength,
	/*out*/ unsigned long int	*signatureCounter)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((clientId == NULL) ||
		(processData == NULL) ||
		(processType == NULL) ||
		(logTime == NULL) ||
		(signatureCounter == NULL) ||
		(signatureValue == NULL) ||
		(signatureValueLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::vector<unsigned char> process_data(processData, processData + processDataLength);
	std::string process_type((const char *)processType, processTypeLength);
	std::string trx_no = LIntegerToString(transactionNumber);
	std::string log_time;
	std::string signature_ctr;
	std::string signature_val;

	std::string process_data_enc = "";

	try
	{ 
		if (process_data.size() != 0) 
		{
			process_data_enc = LEncodeBase64(process_data);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_ENCODING_FAILED"));
		return OTHER_ERROR_ENCODING_FAILED;
	}

	try 
	{
		result = tse_handle->UpdateTransaction(
			client_id,
			trx_no,
			process_data_enc,
			process_type,
			log_time,
			signature_ctr,
			signature_val);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	*signatureCounter = LStringToLongInt(signature_ctr);
	LStringToUtcDateTime(log_time, *logTime);

	std::vector<unsigned char> decoded_stream;
	try
	{
		LDecodeBase64(signature_val, decoded_stream);
	}
	catch (std::runtime_error&) {
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
	}
	*signatureValueLength = (unsigned long) decoded_stream.size();
	*signatureValue = (unsigned char *)TSEEALLOCMEMORY(*signatureValueLength);
	if (*signatureValue == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream, *signatureValue, *signatureValueLength);

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int FinishTransaction(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*in*/  unsigned long int	transactionNumber,
	/*in*/  unsigned char		*processData,
	/*in*/  unsigned long int	processDataLength,
	/*in*/  unsigned char		*processType,
	/*in*/  unsigned long int	processTypeLength,
	/*in*/  unsigned char		*additionalData,
	/*in*/  unsigned long int	additionalDataLength,
	/*out*/ unsigned long int	*logTime,
	/*out*/ unsigned char		**signatureValue,
	/*out*/ unsigned long int	*signatureValueLength,
	/*out*/ unsigned long int	*signatureCounter)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((clientId == NULL) ||
		(processData == NULL) ||
		(processType == NULL) ||
		(additionalData == NULL) ||
		(logTime == NULL) ||
		(signatureCounter == NULL) ||
		(signatureValue == NULL) ||
		(signatureValueLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::vector<unsigned char> process_data(processData, processData + processDataLength);
	std::string process_type((const char *)processType, processTypeLength);
	std::vector<unsigned char> additnl_data(additionalData, additionalData + additionalDataLength);
	std::string trx_no = LIntegerToString(transactionNumber);
	std::string log_time;
	std::string signature_ctr;
	std::string signature_val;

	std::string process_data_enc = "";
	std::string addtnl_data_enc = "";

	try
	{
		if (process_data.size() != 0) 
		{
			process_data_enc = LEncodeBase64(process_data);
		}
		if (additnl_data.size() != 0) 
		{
			addtnl_data_enc = LEncodeBase64(additnl_data);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_ENCODING_FAILED"));
		return OTHER_ERROR_ENCODING_FAILED;
	}

	try 
	{
		result = tse_handle->FinishTransaction(
			client_id,
			trx_no,
			process_data_enc,
			process_type,
			addtnl_data_enc,
			log_time,
			signature_ctr,
			signature_val);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	*signatureCounter = LStringToLongInt(signature_ctr);
	LStringToUtcDateTime(log_time, *logTime);

	std::vector<unsigned char> decoded_stream;
	try
	{
		LDecodeBase64(signature_val, decoded_stream);
	}
	catch (std::runtime_error&) {
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_DECODING_RESULT;
	}
	*signatureValueLength = (unsigned long) decoded_stream.size();
	*signatureValue = (unsigned char *)TSEEALLOCMEMORY(*signatureValueLength);
	if (*signatureValue == NULL)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
		return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
	}
	LVectorToCharArray(decoded_stream, *signatureValue, *signatureValueLength);

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int ExportDataFilteredByTransactionNumberAndClientId(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned long int	transactionNumber,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*out*/ unsigned char		**exportedData,
	/*out*/ unsigned long int	*exportedDataLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((clientId == NULL) ||
		(exportedData == NULL) ||
		(exportedDataLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::string trx_no = LIntegerToString(transactionNumber);
	std::vector<unsigned  char> export_data_all;

	try 
	{
		result = tse_handle->ExportFilteredByTransactionNumber(client_id, trx_no);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	status_code = _ExportData(tse_handle, export_data_all, true, false);
	if ( (status_code != EXECUTION_OK) && (status_code != EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT) )
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	if (export_data_all.size() > 0)
	{
		*exportedDataLength = (unsigned long) export_data_all.size();
		*exportedData = (unsigned char *)TSEEALLOCMEMORY((*exportedDataLength) );
		if (*exportedData == NULL)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
		}

		memcpy_s(*exportedData, *exportedDataLength, export_data_all.data(), *exportedDataLength);
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	if (status_code != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int ExportDataFilteredByTransactionNumber(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned long int	transactionNumber,
	/*out*/ unsigned char		**exportedData,
	/*out*/ unsigned long int	*exportedDataLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((exportedData == NULL) ||
		(exportedDataLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string trx_no = LIntegerToString(transactionNumber);
	std::vector<unsigned  char> export_data_all;

	try 
	{
		result = tse_handle->ExportFilteredByTransactionNumber("", trx_no);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	status_code = _ExportData(tse_handle, export_data_all, true, false);
	if ((status_code != EXECUTION_OK) && (status_code != EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	if (export_data_all.size() > 0)
	{
		*exportedDataLength = (unsigned long) export_data_all.size();
		*exportedData = (unsigned char *)TSEEALLOCMEMORY((*exportedDataLength));
		if (*exportedData == NULL)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
		}

		memcpy_s(*exportedData, *exportedDataLength, export_data_all.data(), *exportedDataLength);
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	
	if (status_code != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;

}

EPSON_TSE_API short int ExportDataFilteredByTransactionNumberIntervalAndClientId(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned long int	startTransactionNumber,
	/*in*/  unsigned long int	endTransactionNumber,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*out*/ unsigned char		**exportedData,
	/*out*/ unsigned long int	*exportedDataLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((clientId == NULL) ||
		(exportedData == NULL) ||
		(exportedDataLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::string start_trx_no = LIntegerToString(startTransactionNumber);
	std::string end_trx_no = LIntegerToString(endTransactionNumber);
	std::vector<unsigned  char> export_data_all;

	try 
	{
		result = tse_handle->ExportFilteredByTransactionNumberInterval(client_id, start_trx_no, end_trx_no);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	status_code = _ExportData(tse_handle, export_data_all, true, false);
	if ((status_code != EXECUTION_OK) && (status_code != EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	if (export_data_all.size() > 0)
	{
		*exportedDataLength = (unsigned long) export_data_all.size();
		*exportedData = (unsigned char *)TSEEALLOCMEMORY((*exportedDataLength));
		if (*exportedData == NULL)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
		}

		memcpy_s(*exportedData, *exportedDataLength, export_data_all.data(), *exportedDataLength);
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	
	if (status_code != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ExportDataFilteredByTransactionNumberInterval(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned long int	startTransactionNumber,
	/*in*/  unsigned long int	endTransactionNumber,
	/*out*/ unsigned char		**exportedData,
	/*out*/ unsigned long int	*exportedDataLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((exportedData == NULL) ||
		(exportedDataLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string start_trx_no = LIntegerToString(startTransactionNumber);
	std::string end_trx_no = LIntegerToString(endTransactionNumber);
	std::vector<unsigned  char> export_data_all;

	try 
	{
		result = tse_handle->ExportFilteredByTransactionNumberInterval("", start_trx_no, end_trx_no);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	status_code = _ExportData(tse_handle, export_data_all, true, false);
	if ((status_code != EXECUTION_OK) && (status_code != EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	if (export_data_all.size() > 0)
	{
		*exportedDataLength = (unsigned long) export_data_all.size();
		*exportedData = (unsigned char *)TSEEALLOCMEMORY((*exportedDataLength));
		if (*exportedData == NULL)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
		}

		memcpy_s(*exportedData, *exportedDataLength, export_data_all.data(), *exportedDataLength);
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif

	if (status_code != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ExportDataFilteredByPeriodOfTimeAndClientId(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned long int	startDate,
	/*in*/  unsigned long int	endDate,
	/*in*/  unsigned char		*clientId,
	/*in*/  unsigned long int	clientIdLength,
	/*out*/ unsigned char		**exportedData,
	/*out*/ unsigned long int	*exportedDataLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((startDate == NULL) ||
		(endDate == NULL) ||
		(clientId == NULL) ||
		(exportedData == NULL) ||
		(exportedDataLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string client_id((const char *)clientId, clientIdLength);
	std::string start_date = LUtcDateTimeToString(startDate);
	std::string end_date = LUtcDateTimeToString(endDate);
	std::vector<unsigned  char> export_data_all;

	try 
	{
		result = tse_handle->ExportFilteredByPeriodOfTime(client_id, start_date, end_date);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	status_code = _ExportData(tse_handle, export_data_all, true, false);
	if ((status_code != EXECUTION_OK) && (status_code != EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	if (export_data_all.size() > 0)
	{
		*exportedDataLength = (unsigned long) export_data_all.size();
		*exportedData = (unsigned char *)TSEEALLOCMEMORY((*exportedDataLength));
		if (*exportedData == NULL)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
		}

		memcpy_s(*exportedData, *exportedDataLength, export_data_all.data(), *exportedDataLength);
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	
	if (status_code != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ExportDataFilteredByPeriodOfTime(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned long int	startDate,
	/*in*/  unsigned long int	endDate,
	/*out*/ unsigned char		**exportedData,
	/*out*/ unsigned long int	*exportedDataLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((startDate == NULL) ||
		(endDate == NULL) ||
		(exportedData == NULL) ||
		(exportedDataLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string start_date = LUtcDateTimeToString(startDate);
	std::string end_date = LUtcDateTimeToString(endDate);
	std::vector<unsigned  char> export_data_all;

	try 
	{
		result = tse_handle->ExportFilteredByPeriodOfTime("", start_date, end_date);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	status_code = _ExportData(tse_handle, export_data_all, true, false);
	if ((status_code != EXECUTION_OK) && (status_code != EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	if (export_data_all.size() > 0)
	{
		*exportedDataLength = (unsigned long) export_data_all.size();
		*exportedData = (unsigned char *)TSEEALLOCMEMORY((*exportedDataLength));
		if (*exportedData == NULL)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
		}

		memcpy_s(*exportedData, *exportedDataLength, export_data_all.data(), *exportedDataLength);
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	
	if (status_code != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ExportData(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ unsigned char		**exportedData,
	/*out*/ unsigned long int	*exportedDataLength)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((exportedData == NULL) ||
		(exportedDataLength == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::vector<unsigned  char> export_data_all;
	std::string tar_size;

	try 
	{
		result = tse_handle->ArchiveExport(tar_size);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	status_code = _ExportData(tse_handle, export_data_all, false, false);
	if ((status_code != EXECUTION_OK) && (status_code != EXECUTION_COMPLETED_BUT_WITH_UNFINALIZED_EXPORT))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	if (export_data_all.size() > 0)
	{
		*exportedDataLength = (unsigned long) export_data_all.size();
		*exportedData = (unsigned char *)TSEEALLOCMEMORY((*exportedDataLength));
		if (*exportedData == NULL)
		{
			TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT"));
			return EXECUTION_COMPLETED_BUT_WITH_ERROR_ALLOCATING_FOR_RESULT;
		}

		memcpy_s(*exportedData, *exportedDataLength, export_data_all.data(), *exportedDataLength);
	}

#if (PRINT_MEMORY == true)
	TSEMemoryPrintAllocated(tse_log);
#endif
	
	if (status_code != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("ExportData=%d", status_code));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int FinalizeExport(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  bool				deleteStoredData)
{
	short int status_code;
	std::string result;
	std::string delete_stored_data = (deleteStoredData) ? "true" : "false";

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try 
	{
		result = tse_handle->FinalizeExport(delete_stored_data);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}
	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int CancelExport(
	/*in*/  EPSON_TSE_HANDLE	tseHandle)
{
	short int status_code;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	try
	{
		result = tse_handle->CancelExport();
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}

EPSON_TSE_API short int ChangePuk(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*oldPuk,
	/*in*/  unsigned long int	oldPukLength,
	/*in*/  unsigned char		*newPuk,
	/*in*/  unsigned long int	newPukLength,
	/*in*/  unsigned char		*secretKey,
	/*in*/  unsigned long int	secretKeyLength,
	/*out*/ short int			*remainingRetries)
{
	short int status_code, status_code_m;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((oldPuk == NULL) ||
		(newPuk == NULL) ||
		(remainingRetries == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string old_puk((const char *)oldPuk, oldPukLength);
	std::string new_puk((const char *)newPuk, newPukLength);
	std::string secret_key((const char *)secretKey, secretKeyLength);
	std::string remaining_retries;
	std::string challenge;
	std::string hash = "";
	std::string cha_sctkey_s = "";
	*remainingRetries = -1;

	try
	{
		result = tse_handle->GetChallenge(EPSON_TSE_ADMIN, challenge);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}
	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	cha_sctkey_s = challenge + secret_key;
	std::vector<unsigned char> cha_sctkey_v(cha_sctkey_s.begin(), cha_sctkey_s.end());

	try
	{
		hash = LEncodeBase64Sha256(cha_sctkey_v);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_ENCODING_FAILED"));
		return OTHER_ERROR_ENCODING_FAILED;
	}

	try
	{
		result = tse_handle->AuthenticateHost(EPSON_TSE_ADMIN, hash);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	try
	{
		result = tse_handle->ChangePuk(EPSON_TSE_ADMIN, old_puk, new_puk, remaining_retries);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	status_code_m = LGetStatusCode(result);
	if (status_code_m == TSE1_ERROR_AUTHENTICATION_FAILED)
	{
		*remainingRetries = LStringToShortInt(remaining_retries);
	}

	try
	{
		result = tse_handle->DeauthenticateHost(EPSON_TSE_ADMIN);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if (status_code_m != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%d", status_code_m));
		return status_code_m;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API short int UnblockUser(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned char		*userId,
	/*in*/  unsigned long int	userIdLength,
	/*in*/  unsigned char		*puk,
	/*in*/  unsigned long int	pukLength,
	/*in*/  unsigned char		*newPin,
	/*in*/  unsigned long int	newPinLength,
	/*in*/  unsigned char		*secretKey,
	/*in*/  unsigned long int	secretKeyLength,
	/*out*/ short int			*remainingRetries)
{
	short int status_code, status_code_m;
	std::string result;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return DEVICE_NO_CONNECTION;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN"));
	if ((userId == NULL) ||
		(puk == NULL) ||
		(newPin == NULL) ||
		(secretKey == NULL))
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return OTHER_ERROR_INVALID_POINTER;
	}

	std::string user_id((const char *)userId, userIdLength);
	std::string puk_s((const char *)puk, pukLength);
	std::string new_pin((const char *)newPin, newPinLength);
	std::string secret_key((const char *)secretKey, secretKeyLength);
	std::string challenge;
	std::string hash = "";
	std::string cha_sctkey_s = "";
	std::string remaining_retries;
	*remainingRetries = -1;

	try 
	{
		result = tse_handle->GetChallenge(EPSON_TSE_ADMIN, challenge);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}
	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	cha_sctkey_s = challenge + secret_key;
	std::vector<unsigned char> cha_sctkey_v(cha_sctkey_s.begin(), cha_sctkey_s.end());

	try
	{
		hash = LEncodeBase64Sha256(cha_sctkey_v);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_ENCODING_FAILED"));
		return OTHER_ERROR_ENCODING_FAILED;
	}
	
	try 
	{
		result = tse_handle->AuthenticateHost(EPSON_TSE_ADMIN, hash);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}
	
	try 
	{
		if (user_id == EPSON_TSE_ADMIN)
		{
			result = tse_handle->UnblockUserForAdmin(EPSON_TSE_ADMIN, puk_s, new_pin, remaining_retries);
		}
		else
		{
			result = tse_handle->UnblockUserForTimeAdmin(EPSON_TSE_ADMIN, puk_s, new_pin, remaining_retries);
		}
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}
	status_code_m = LGetStatusCode(result);
	if (status_code_m == TSE1_ERROR_AUTHENTICATION_FAILED)
	{
		*remainingRetries = LStringToShortInt(remaining_retries);
	}

	try
	{
		result = tse_handle->DeauthenticateHost(EPSON_TSE_ADMIN);
	}
	catch (std::runtime_error&)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("DEVICE_CONNECTION_ERROR"));
		return DEVICE_CONNECTION_ERROR;
	}

	if (status_code_m != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%d", status_code_m));
		return status_code_m;
	}

	if ((status_code = LGetStatusCode(result)) != EXECUTION_OK)
	{
		TSEPRINTLOG(tse_log, LIBLOG_LEVEL_ERROR, ("OperateError=%s", result.c_str()));
		return status_code;
	}

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return EXECUTION_OK;
}


EPSON_TSE_API void TSEMemDealloc(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ void				*memoryAddress)
{
	TSEMemoryDeallocate(memoryAddress);
#if (PRINT_MEMORY == true)
	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("HANDLE [0x%x]", tseHandle));
		TSEMemoryPrintAllocated(g_log);

	}
	else
	{
		TSEOperate * tse_handle = (TSEOperate *)tseHandle;
		TSELogger& tse_log = tse_handle->GetLoggerIntance();

		TSEMemoryPrintAllocated(tse_log);
	}
#endif
	return;
}

EPSON_TSE_API void TSEMemDeallocTseInfoObject(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*out*/ EPSON_TSE_INFO		*tseInfo)
{
	TSEMemoryDeallocateTseInfoObject(tseInfo);

#if (PRINT_MEMORY == true)
	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION - INVALID HANDLE [0x%x]", tseHandle));
		TSEMemoryPrintAllocated(g_log);
		
	}
	else
	{
		TSEOperate * tse_handle = (TSEOperate *)tseHandle;
		TSELogger& tse_log = tse_handle->GetLoggerIntance();

		TSEMemoryPrintAllocated(tse_log);
	}
#endif
	return;
}

EPSON_TSE_API void TSESetLog(
	/*in*/  EPSON_TSE_HANDLE	tseHandle,
	/*in*/  unsigned short int	logLevel,
	/*in*/  unsigned short int	logMethod)
{
	
	uint8_t log_method = (uint8_t)logMethod;
	LiblogLevel log_level = (LiblogLevel)logLevel;

	if (IsTseHandleValid(tseHandle) == false)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("DEVICE_NO_CONNECTION"));
		return;
	}

	TSEOperate * tse_handle = (TSEOperate *)tseHandle;
	TSELogger& tse_log = tse_handle->GetLoggerIntance();

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("IN l=%d m=%d", logLevel, logMethod));

	tse_log.SetLogLevel(log_level, log_method);

	TSEPRINTLOG(tse_log, LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return;
}

EPSON_TSE_API void TSEGetErrorDescription(
	/*in*/  short int			errorCode,
	/*in*/  short int			language,
	/*out*/ unsigned char		errorDescription[ERROR_DESC_SIZE])
{
	//TSEPRINTLOG(LIBLOG_LEVEL_TRACE, ("IN"));
	if (errorDescription == NULL)
	{
		TSEPRINTLOG(g_log, LIBLOG_LEVEL_ERROR, ("OTHER_ERROR_INVALID_POINTER"));
		return;
	}

	char * p_ed = (char *)errorDescription;
	std::string s_err_desc = "";

	memset(p_ed, '\0', ERROR_DESC_SIZE);

	if (language == ERROR_DESC_DE)
	{
		try
		{
			s_err_desc = gErrDescDE.at(errorCode);
		}
		catch (std::out_of_range&)
		{
			s_err_desc = "Unbekannter Fehler.";
		}
	}
	else
	{
		try
		{
			s_err_desc = gErrDescEN.at(errorCode);
		}
		catch (std::out_of_range&)
		{
			s_err_desc = "Unknown Error.";
		}
	}

	strcpy_s(p_ed, (ERROR_DESC_SIZE - 1), s_err_desc.c_str());
	//TSEPRINTLOG(LIBLOG_LEVEL_TRACE, ("OUT OK"));
	return;
}

/*EOF*/