// Copyright 2019-2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/09  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#define WIN32_LEAN_AND_MEAN
#define _WINSOCK_DEPRECATED_NO_WARNINGS

#include "AccessBySocket.h"
#include "ENPC_ForceSend.h"
#include "TSEPrintLog.h"
#include <stdexcept>
#include <ws2tcpip.h>
// Need to link with Ws2_32.lib, Mswsock.lib, and Advapi32.lib
#pragma comment (lib, "Ws2_32.lib")
//#pragma comment (lib, "Mswsock.lib")
//#pragma comment (lib, "AdvApi32.lib")

#define XML_RCV_SIZE 1024


AccessSocket::AccessSocket(const char * tIpAddr, const char * tPort, bool isEposDevXml)
{
	mIpAddr = tIpAddr;
	mPort = tPort;
	mIsEposDevXml = isEposDevXml;
	mSocket = INVALID_SOCKET;
	Connect();
}

AccessSocket::~AccessSocket()
{
	int nResult = 0;

	if (mSocket == INVALID_SOCKET)
	{
		return;
	}

	nResult = shutdown(mSocket, SD_SEND);
	if (nResult == SOCKET_ERROR) 
	{
		TSEPRINTLOG(LIBLOG_LEVEL_WARNING, ("socket error=%d", WSAGetLastError()));
	}

	closesocket(mSocket);
	mSocket = INVALID_SOCKET;
	WSACleanup();
}

void AccessSocket::Connect() 
{
	int nResult;
	struct addrinfo *result = NULL,
		*ptr = NULL,
		hints;
	WSADATA wsaData;

	if (mSocket != INVALID_SOCKET) 
	{
		shutdown(mSocket, SD_SEND);
		closesocket(mSocket);
		mSocket = INVALID_SOCKET;
		WSACleanup();
	}

	nResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
	if (nResult != 0) 
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket error=%d", WSAGetLastError()));
		throw 10;
	}

	ZeroMemory(&hints, sizeof(hints));
	hints.ai_family = AF_INET;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = IPPROTO_TCP;

	nResult = getaddrinfo(mIpAddr.c_str(), mPort.c_str(), &hints, &result);
	if (nResult != 0) 
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket error=%d", WSAGetLastError()));
		WSACleanup();
		throw 11;
	}

	for (ptr = result; ptr != NULL; ptr = ptr->ai_next) 
	{
		mSocket = socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol);

		if (mSocket == INVALID_SOCKET) 
		{
			TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket error=%d", WSAGetLastError()));
			WSACleanup();
			throw 12;
		}

		nResult = connect(mSocket, ptr->ai_addr, (int)ptr->ai_addrlen);

		if (nResult == SOCKET_ERROR) 
		{
			TSEPRINTLOG(LIBLOG_LEVEL_WARNING, ("socket error=%d", WSAGetLastError()));			
			closesocket(mSocket);
			mSocket = INVALID_SOCKET;
			continue;
		}
		break;
	}

	freeaddrinfo(result);

	if (mSocket == INVALID_SOCKET) 
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket error=INVALID_SOCKET"));
		WSACleanup();
		throw 13;
	}

	u_long nonblock = 1;
	ioctlsocket(mSocket, FIONBIO, &nonblock);

}

void AccessSocket::SocketSend(const std::vector<BYTE>& tBuffer)
{
	int nResult = SOCKET_ERROR;

	if (!SocketIsConnected()) {
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket disconnected"));
		throw 21;
	}

	const BYTE * tempBuf;
	unsigned long remByteToSend = (unsigned long) tBuffer.size();
	unsigned long totalByteSent = 0;

	while (remByteToSend > 0)
	{
		tempBuf = tBuffer.data() + totalByteSent;
		nResult = send(mSocket, (const char *)tempBuf, remByteToSend, 0);

		if (nResult == SOCKET_ERROR)
		{
			int wserror = WSAGetLastError();

			if (wserror != WSAEWOULDBLOCK) 
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket disconnected"));
				throw 22;
			}

			fd_set write_set;
			FD_ZERO(&write_set);
			FD_SET(mSocket, &write_set);

			struct timeval time_out;
			time_out.tv_sec = 120;
			time_out.tv_usec = 0;

			int result = select(0, NULL, &write_set, NULL, &time_out);
			if (result != 1)
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket disconnected"));
				throw 23;
			}
		}//if nResult
		else 
		{
			remByteToSend -= nResult;
			totalByteSent += nResult;
		}
	}	
}

void AccessSocket::SocketRecv(std::vector<BYTE>& tBuffer, const unsigned long& tMaxSize, const int& tTimeout)
{
	int byteRcvd = 0;
	unsigned long byteRemain = tMaxSize;

	if (!SocketIsConnected())
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket disconnected"));
		throw 31;
	}

	tBuffer.clear();

	int nSel = 0;
	struct timeval tv;
	tv.tv_usec = 0;
	tv.tv_sec = (tTimeout / 1000);

	fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(mSocket, &rfds);

	char *rcvBuf = NULL;

	try
	{
		rcvBuf = new char[(mIsEposDevXml) ? XML_RCV_SIZE : tMaxSize];
	}
	catch (std::bad_alloc &) 
	{ 
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket error=ALLOC"));
		throw 34; // std::runtime_error("Unable to allocate memory.");
	}

	while (true)
	{
		nSel = select((int)(mSocket + 1), &rfds, NULL, NULL, &tv);
		if (nSel > 0)
		{
			if ((byteRcvd = recv(mSocket, rcvBuf, (mIsEposDevXml ? XML_RCV_SIZE : byteRemain), 0)) > 0)
			{
				tBuffer.insert(tBuffer.end(), rcvBuf, rcvBuf + byteRcvd);
				byteRemain -= byteRcvd;

				if (mIsEposDevXml) //for epos-device xml check
				{
					if (rcvBuf[byteRcvd - 1] == 0x00)
					{
						break;
					}
				}
				else
				{
					if (byteRemain == 0) 
					{
						break;
					}
				}
			}
			else
			{
				delete rcvBuf;
				return;// throw 33;
			}
		}
		else if (nSel == SOCKET_ERROR)
		{
			delete rcvBuf;
			TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket disconnected"));
			throw 32;
		}
	}
	delete rcvBuf;
}

bool AccessSocket::SocketIsConnected()
{
	if (mSocket == INVALID_SOCKET)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket disconnected. Invalid Socket"));
		return FALSE;
	}

	char data;

	int nResult = 0;
	Sleep(10);

	nResult = recv(mSocket, &data, 1, MSG_PEEK );//check one byte
	if (nResult == SOCKET_ERROR) 
	{
		int  wserror = WSAGetLastError();
		if (wserror != WSAEWOULDBLOCK) { //if (wserror == WSAECONNRESET)
			TSEPRINTLOG(LIBLOG_LEVEL_WARNING, ("socket disconnected=%d", wserror));
			if (mIsEposDevXml) return false;
			try 
			{
				Connect(); //Try to reconnect
			}
			catch (int &) 
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket disconnected. Could not reconnect"));
				return false; 
			}
		}
	}

	return true;
}

void AccessSocket::EnableForceSend(const char * tIpAddr)
{
	unsigned long ipv4addr;
	if (InetPtonWr(AF_INET, tIpAddr, &ipv4addr))
	{
		ENPC_ForceSend(ipv4addr, true);
	}
}

int  AccessSocket::InetPtonWr(int tAf, const char *tSrc, void *tDst)
{
	struct sockaddr_storage ss;
	int ss_size = sizeof(ss);
	char tmpSrc[INET6_ADDRSTRLEN + 1];

	ZeroMemory(&ss, sizeof(ss));
	strncpy_s(tmpSrc, tSrc, INET6_ADDRSTRLEN + 1);
	tmpSrc[INET6_ADDRSTRLEN] = 0;

	if (WSAStringToAddress(tmpSrc, tAf, NULL, (struct sockaddr *)&ss, &ss_size) != 0)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket error=%d", WSAGetLastError()));
		return 0; //NG
	}
		
	switch (tAf) 
	{
		case AF_INET:
			*(struct in_addr *) tDst = ((struct sockaddr_in *)&ss)->sin_addr;
			break;
		case AF_INET6:
			*(struct in6_addr *)tDst = ((struct sockaddr_in6 *)&ss)->sin6_addr;
			break;
		default: 
		{
			TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("socket error"));
			return 0;
		}
	}

	return 1; //Successful
}

//EOF