// Copyright 2019-2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/12  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include <sstream>
#include <iostream>
#include "TSEOperate.h"
#include "TSEOptions.h"
#include "TSESimpleJsonAndXmlParser.h"
#include "TSEPrintLog.h"
#include "TSEConfig.h"
#include "TSEAccessByEposDevice.h"
#include "TSEAccessByEscPosNetwork.h"
#include "TSEAccessByEscPosSerial.h"
#ifdef EPSON_TSE_PCS_SUPPORT
#include "TSEAccessByEscPosPCS.h"
#endif /*EPSON_TSE_PCS_SUPPORT*/
#include <windows.h>

#define TSE_ACCESS_RETRY		25
#define TSE_CONNECT_RETRY		5

#define AS_STRING		0x01
#define AS_NONSTRING	0x02
#define AS_ARRAY		0x03

TSEOperate::TSEOperate(const TSEOptions& tTseOpt) 
{	
	int retry = 0;
	std::stringstream ssJsonH, ssJsonF;

	mConMethod = tTseOpt.mMethod;
	mTseVendor = "TSE1";
	mTseAcs = NULL;
	mIsTseDeviceOpen = false;

	unsigned short connectRetry = 0;
	if (!connectRetry)
	{
		TSEConfig tseConfig;
		if (tseConfig.GetInitConnectRetry(connectRetry) == false)
		{
			connectRetry = TSE_CONNECT_RETRY;
		}
	}

	while (retry < connectRetry)
	{
		try 
		{
			switch (tTseOpt.mMethod)
			{
			case ACCESS_BY_NETWORK_ESCPOS:
				mTseAcs = new TSEAccessByEscPosNetwork(tTseOpt.mIdentifier); break;
			case ACCESS_BY_EPOS_DEVICE:
				mTseAcs = new TSEAccessByEposDevice(tTseOpt.mIdentifier); break;
#ifdef EPSON_TSE_PCS_SUPPORT				
			case ACCESS_BY_PCS_ESCPOS:
				mTseAcs = new TSEAccessByEscPosPCS(tTseOpt.mIdentifier); break;
#endif /*EPSON_TSE_PCS_SUPPORT*/
			case ACCESS_BY_SERIAL_ESCPOS:
			default:
				mConMethod = ACCESS_BY_SERIAL_ESCPOS;
				mTseAcs = new TSEAccessByEscPosSerial(tTseOpt.mIdentifier); break;
			}
		}
		catch (std::runtime_error&)
		{
			TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Create Error. Retrying..."));
			mTseAcs = NULL;
		}

		if (mTseAcs != NULL)
		{
			break;
		}
		retry++;
	}

	if (mTseAcs == NULL) 
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Failed to create port."));
		throw std::runtime_error("");
	}

	mTseAcs->SetTseDeviceId(tTseOpt.mTseHandler);

	ssJsonH << "\"storage\": {"
		<< "\"type\": \"TSE\","
		<< "\"vendor\" : \"" << mTseVendor << "\""
		<< "},";

	ssJsonF << "\"compress\" : {"
		<< "\"required\": false,"
		<< "\"type\" : \"\""
		<< "}";

	mJsonHeader = ssJsonH.str();
	mJsonFooter = ssJsonF.str();
}

TSEOperate::~TSEOperate() 
{
	delete mTseAcs;
	mTseAcs = NULL;

	unsigned long logFileSize = 0;
	TSEConfig tseConfig;
	if (tseConfig.GetLogFileSize(logFileSize) == true)
	{
		TSEPrintLog::ManageLogFile(logFileSize);
	}
}

unsigned long TSEOperate::GetTotalTime()
{
	return mTseAcs->GetTotalTime();
}

unsigned long TSEOperate::GetLastRcvTime() 
{
	return mTseAcs->GetLastRcvTime();
}

void TSEOperate::TSEOpenDevice()
{
	int retry = 0;
	while (retry < TSE_ACCESS_RETRY)
	{
		if (mTseAcs->EnableTseAccess())
		{
			break;
		}
		retry++;
		Sleep(1000);
	}

	if (retry == TSE_ACCESS_RETRY)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Failed to enable TSE after several retries."));
		throw std::runtime_error("");
	}

	mIsTseDeviceOpen = true;
}

void TSEOperate::TSECloseDevice()
{
	int retry = 0;
	while (retry < TSE_ACCESS_RETRY)
	{
		if (mTseAcs->DisableTseAccess())
		{
			break;
		}
		retry++;
		Sleep(1000);
	}

	if (retry == TSE_ACCESS_RETRY)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Failed to disable TSE after several retries."));
		throw std::runtime_error("");
	}

	mIsTseDeviceOpen = false; 
}


std::string TSEOperate::GetChallenge(
	const std::string& tUserId, 
	std::string& tChallenge )
{
	std::string opfunction = "GetChallenge";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tUserId << "\""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction) 
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tChallenge = GetValueFromJsonKey(jsonOutput, "challenge", AS_STRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::AuthenticateUserForAdmin(
	const std::string& tUserId,
	const std::string& tPin,
	const std::string& tHash,
	std::string& tRemainingRetries)
{
	std::string opfunction = "AuthenticateUserForAdmin";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;
	
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tUserId << "\","
		<< "\"pin\":\"" << tPin << "\","
		<< "\"hash\":\"" << tHash << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRemainingRetries = GetValueFromJsonKey(jsonOutput, "remainingRetries", AS_NONSTRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);

}

std::string TSEOperate::AuthenticateUserForTimeAdmin(
	const std::string& tClientId,
	const std::string& tPin,
	const std::string& tHash,
	std::string& tRemainingRetries)
{
	std::string opfunction = "AuthenticateUserForTimeAdmin";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\","
		<< "\"pin\":\"" << tPin << "\","
		<< "\"hash\":\"" << tHash << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRemainingRetries = GetValueFromJsonKey(jsonOutput, "remainingRetries", AS_NONSTRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);

}

std::string TSEOperate::AuthenticateHost(
	const std::string& tUserId,
	const std::string& tHash)
{
	std::string opfunction = "AuthenticateHost";
	std::string jsonResult = "";
	std::stringstream ssJson;


	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tUserId << "\","
		<< "\"hash\":\"" << tHash << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}


std::string TSEOperate::LogOutForAdmin()
{
	std::string opfunction = "LogOutForAdmin";
	std::string jsonResult = "";
	std::stringstream ssJson;
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::LogOutForTimeAdmin(
	const std::string& tClientId)
{
	std::string opfunction = "LogOutForTimeAdmin";
	std::string jsonResult = "";
	std::stringstream ssJson;
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::DeauthenticateHost(
	const std::string& tUserId)
{
	std::string opfunction = "DeauthenticateHost";
	std::string jsonResult = "";
	std::stringstream ssJson;


	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tUserId << "\""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::UpdateTimeForFirst(
	const std::string& tClientId, 
	const std::string& tNewTime,
	const std::string& tUseTimeSync)
{
	std::string opfunction = "UpdateTimeForFirst";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tClientId << "\","
		<< "\"newDateTime\":\"" << tNewTime << "\","
		<< "\"useTimeSync\":" << tUseTimeSync 
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::UpdateTime(
	const std::string& tClientId, 
	const std::string& tNewTime,
	const std::string& tUseTimeSync)
{
	std::string opfunction = "UpdateTime";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tClientId << "\","
		<< "\"newDateTime\":\"" << tNewTime << "\","
		<< "\"useTimeSync\":"<< tUseTimeSync 
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::StartTransaction(
	const std::string& tClientId,
	const std::string& tProcessData,
	const std::string& tProcessType,
	const std::string& tAdditionalData,
	std::string& tTrxNo,
	std::string& tLogTime,
	std::string& tSerialNo,
	std::string& tSignatureCtr,
	std::string& tSignature)
{
	std::string opfunction = "StartTransaction";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::string tempStr = "";

	std::stringstream ssJson, ssJsonRet1, ssJsonRet2;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\","
		<< "\"processData\":\"" << tProcessData << "\","
		<< "\"processType\":\"" << tProcessType << "\","
		<< "\"additionalData\":\"" << tAdditionalData << "\""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");

	tempStr = GetValueFromJsonKey(jsonOutput, "transactionNumber", AS_NONSTRING);
	ssJsonRet1 << tempStr;
	ssJsonRet1 >> tTrxNo;

	tempStr = GetValueFromJsonKey(jsonOutput, "signatureCounter", AS_NONSTRING);
	ssJsonRet2 << tempStr;
	ssJsonRet2 >> tSignatureCtr;

	tLogTime = GetValueFromJsonKey(jsonOutput, "logTime", AS_STRING);
	tSerialNo = GetValueFromJsonKey(jsonOutput, "serialNumber", AS_STRING);
	tSignature = GetValueFromJsonKey(jsonOutput, "signature", AS_STRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::UpdateTransaction(
	const std::string& tClientId,
	const std::string& tTrxNo,
	const std::string& tProcessData,
	const std::string& tProcessType,
	std::string& tLogTime,
	std::string& tSignatureCtr,
	std::string& tSignature)
{
	std::string opfunction = "UpdateTransaction";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::string tempStr = "";

	std::stringstream ssJson, ssJsonRet1;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\","
		<< "\"transactionNumber\":" << tTrxNo << ","
		<< "\"processData\":\"" << tProcessData << "\","
		<< "\"processType\":\"" << tProcessType << "\""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");

	tempStr = GetValueFromJsonKey(jsonOutput, "signatureCounter", AS_NONSTRING);
	ssJsonRet1 << tempStr;
	ssJsonRet1 >> tSignatureCtr;

	tLogTime = GetValueFromJsonKey(jsonOutput, "logTime", AS_STRING);
	tSignature = GetValueFromJsonKey(jsonOutput, "signature", AS_STRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::FinishTransaction(
	const std::string& tClientId,
	const std::string& tTrxNo,
	const std::string& tProcessData,
	const std::string& tProcessType,
	const std::string& tAdditionalData,
	std::string& tLogTime,
	std::string& tSignatureCtr,
	std::string& tSignature)
{
	std::string opfunction = "FinishTransaction";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::string tempStr = "";

	std::stringstream ssJson, ssJsonRet1;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\","
		<< "\"transactionNumber\":" << tTrxNo << ","
		<< "\"processData\":\"" << tProcessData << "\","
		<< "\"processType\":\"" << tProcessType << "\","
		<< "\"additionalData\":\"" << tAdditionalData << "\""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");

	tempStr = GetValueFromJsonKey(jsonOutput, "signatureCounter", AS_NONSTRING);
	ssJsonRet1 << tempStr;
	ssJsonRet1 >> tSignatureCtr;

	tLogTime = GetValueFromJsonKey(jsonOutput, "logTime", AS_STRING);
	tSignature = GetValueFromJsonKey(jsonOutput, "signature", AS_STRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}


std::string TSEOperate::ExportFilteredByTransactionNumber(
	const std::string& tUserId,
	const std::string& tTrxNo)
{
	std::string opfunction = "ExportFilteredByTransactionNumber";
	std::string jsonResult = "";
	std::string jsonOutput = "";

	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tUserId << "\","
		<< "\"transactionNumber\":" << tTrxNo << ""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::ExportFilteredByTransactionNumberInterval(
	const std::string& tUserId,
	const std::string& tStartTrxNo,
	const std::string& tEndTrxNo)
{
	std::string opfunction = "ExportFilteredByTransactionNumberInterval";
	std::string jsonResult = "";
	std::string jsonOutput = "";

	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tUserId << "\","
		<< "\"startTransactionNumber\":" << tStartTrxNo << ","
		<< "\"endTransactionNumber\":" << tEndTrxNo << ""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);
	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::ExportFilteredByPeriodOfTime(
	const std::string& tUserId,
	const std::string& tStartDateTime,
	const std::string& tEndDateTime)
{
	std::string opfunction = "ExportFilteredByPeriodOfTime";
	std::string jsonResult = "";
	std::string jsonOutput = "";

	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tUserId << "\","
		<< "\"startDate\":\"" << tStartDateTime << "\","
		<< "\"endDate\":\"" << tEndDateTime << "\""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);
	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::ArchiveExport(
	std::string& tTarSize)
{
	std::string opfunction = "ArchiveExport";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::string tempStr = "";

	std::stringstream ssJson, ssJsonRet1;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");

	tempStr = GetValueFromJsonKey(jsonOutput, "tarExportSize", AS_NONSTRING);
	ssJsonRet1 << tempStr;
	ssJsonRet1 >> tTarSize;

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetExportData(
	std::string& tExportData,
	std::string& tExportStatus)
{
	std::string opfunction = "GetExportData";
	std::string jsonOutput = "";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< "\"compress\" : {"
		<< "\"required\": false,"
		<< "\"type\" : \"zip_deflate\""
		<< "}"
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tExportData = GetValueFromJsonKey(jsonOutput, "exportData", AS_STRING);
	tExportStatus = GetValueFromJsonKey(jsonOutput, "exportStatus", AS_STRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}


std::string TSEOperate::FinalizeExport(
	const std::string& tDeleteData)
{

	std::string opfunction = "FinalizeExport";
	std::string jsonOutput = "";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"deleteData\":" << tDeleteData << ""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::CancelExport()
{
	std::string opfunction = "CancelExport";
	std::string jsonOutput = "";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::Setup(
	const std::string& tPuk,
	const std::string& tAdmPin,
	const std::string& tTimeAdmPin)
{
	std::string opfunction = "SetUp";
	std::string jsonResult = "";
	std::stringstream ssJson;
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"puk\":\"" << tPuk << "\","
		<< "\"adminPin\":\"" << tAdmPin << "\","
		<< "\"timeAdminPin\":\"" << tTimeAdmPin << "\""
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::RegisterSecretKey(
	const std::string& tNewSecretKey)
{
	std::string opfunction = "RegisterSecretKey";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"secretKey\":\"" << tNewSecretKey << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::DisableSecureElement()
{
	std::string opfunction = "DisableSecureElement";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::UnblockUserForAdmin(
	const std::string& tUserId,
	const std::string& tPuk,
	const std::string& tNewPin,
	std::string& tRemainingRetries)
{
	std::string opfunction = "UnblockUserForAdmin";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tUserId << "\","
		<< "\"puk\":\"" << tPuk << "\","
		<< "\"newPin\":\"" << tNewPin << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRemainingRetries = GetValueFromJsonKey(jsonOutput, "remainingRetries", AS_NONSTRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::UnblockUserForTimeAdmin(
	const std::string& tUserId,
	const std::string& tPuk,
	const std::string& tNewPin,
	std::string& tRemainingRetries)
{
	std::string opfunction = "UnblockUserForTimeAdmin";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tUserId << "\","
		<< "\"puk\":\"" << tPuk << "\","
		<< "\"newPin\":\"" << tNewPin << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRemainingRetries = GetValueFromJsonKey(jsonOutput, "remainingRetries", AS_NONSTRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::ChangePuk(
	const std::string& tUserId,
	const std::string& tOldPuk,
	const std::string& tNewPuk,
	std::string& tRemainingRetries
)
{
	std::string opfunction = "ChangePuk";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tUserId << "\","
		<< "\"oldPuk\":\"" << tOldPuk << "\","
		<< "\"newPuk\":\"" << tNewPuk << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRemainingRetries = GetValueFromJsonKey(jsonOutput, "remainingRetries", AS_NONSTRING);
	
	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::ChangePinForAdmin(
	const std::string& tClientId,
	const std::string& tOldPin,
	const std::string& tNewPin,
	std::string& tRemainingRetries
)
{
	std::string opfunction = "ChangePinForAdmin";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userId\":\"" << tClientId << "\","
		<< "\"oldPin\":\"" << tOldPin << "\","
		<< "\"newPin\":\"" << tNewPin << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRemainingRetries = GetValueFromJsonKey(jsonOutput, "remainingRetries", AS_NONSTRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}


std::string TSEOperate::ChangePinForTimeAdmin(
	const std::string& tUserId,
	const std::string& tOldPin,
	const std::string& tNewPin,
	std::string& tRemainingRetries
)
{
	std::string opfunction = "ChangePinForTimeAdmin";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tUserId << "\","
		<< "\"oldPin\":\"" << tOldPin << "\","
		<< "\"newPin\":\"" << tNewPin << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRemainingRetries = GetValueFromJsonKey(jsonOutput, "remainingRetries", AS_NONSTRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::RegisterClient(
	const std::string& tClientId)
{
	std::string opfunction = "RegisterClient";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::DeregisterClient(
	const std::string& tClientId)
{
	std::string opfunction = "DeregisterClient";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::UnlockTSE()
{
	std::string opfunction = "UnlockTSE";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::LockTSE()
{
	std::string opfunction = "LockTSE";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetAuthenticatedUserList(
	const std::string& tUserRole, 
	std::string & tAuthenticatedUserList)
{
	std::string opfunction = "GetAuthenticatedUserList";
	std::string jsonResult = "";

	std::string jsonOutput = "";
	std::stringstream ssJson;
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"userRole\":\"" << tUserRole << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tAuthenticatedUserList = GetValueFromJsonKey(jsonOutput, "authenticatedUserList", AS_ARRAY);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetLogMessageCertificate(
	std::string & tLogMsgCertificate)
{
	std::string opfunction = "GetLogMessageCertificate";
	std::string jsonResult = "";

	std::string jsonOutput = "";
	std::stringstream ssJson;
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tLogMsgCertificate = GetValueFromJsonKey(jsonOutput, "logMessageCertificate", AS_STRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetLastTransactionResponse(
	const std::string& tClientId, 
	std::string& tTrxNo,
	std::string& tLogTime,
	std::string& tSignatureCtr,
	std::string& tSignature,
	std::string& tSerialNo)
{
	std::string opfunction = "GetLastTransactionResponse";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::string tempStr = "";
	std::stringstream ssJson;
	std::stringstream ssJsonRet1, ssJsonRet2;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");

	tempStr = GetValueFromJsonKey(jsonOutput, "transactionNumber", AS_NONSTRING);
	ssJsonRet1 << tempStr;
	ssJsonRet1 >> tTrxNo;

	tempStr = GetValueFromJsonKey(jsonOutput, "signatureCounter", AS_NONSTRING);
	ssJsonRet2 << tempStr;
	ssJsonRet2 >> tSignatureCtr;

	tLogTime = GetValueFromJsonKey(jsonOutput, "logTime", AS_STRING);
	tSerialNo = GetValueFromJsonKey(jsonOutput, "serialNumber", AS_STRING);
	tSignature = GetValueFromJsonKey(jsonOutput, "signature", AS_STRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetStartedTransactionList(
	const std::string& tClientId,
	std::string& tStartedTrList)
{
	std::string opfunction = "GetStartedTransactionList";
	std::string jsonOutput = "";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "\"clientId\":\"" << tClientId << "\""
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tStartedTrList = GetValueFromJsonKey(jsonOutput, "startedTransactionNumberList", AS_ARRAY);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetRegisteredClientList(
	std::string& tRegisteredClientList)
{
	std::string opfunction = "GetRegisteredClientList";
	std::string jsonResult = "";

	std::string jsonOutput = "";
	std::stringstream ssJson;
	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tRegisteredClientList = GetValueFromJsonKey(jsonOutput, "registeredClientIdList", AS_ARRAY);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}


std::string TSEOperate::GetStorageInfo(
	std::string& tTseInfo)
{
	std::string opfunction = "GetStorageInfo";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson	<< "{"
				<< mJsonHeader
				<< "\"function\" : \"" << opfunction << "\","
				<< "\"input\" : {"
				<< "},"
				<< mJsonFooter
			<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	tTseInfo = GetJsonObj(jsonResult, "tseInformation");

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetStorageSmartInfo(
	std::string& tSmartInfo)
{
	std::string opfunction = "GetStorageInfo";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";

	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	tSmartInfo = GetJsonObj(jsonResult, "smartInformation");

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::FactoryReset()
{
	std::string opfunction = "FactoryReset";
	std::string jsonResult = "";
	std::stringstream ssJson;
	ssJson	<< "{"
				<< mJsonHeader
				<< "\"function\" : \"" << opfunction << "\","
				<< "\"input\" : {"
				<< "},"
				<< mJsonFooter
			<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::SetTimeOutInterval(const std::string& tAdmin, const std::string& tTimeAdmin, const std::string& tExport)
{
	std::string opfunction = "SetTimeOutInterval";
	std::string jsonResult = "";
	std::stringstream ssJson;

	ssJson	<< "{"
				<< mJsonHeader
				<< "\"function\" : \"" << opfunction << "\","
				<< "\"input\" : {"
					<< "\"timeoutIntervalForAdmin\":" << tAdmin << ","
					<< "\"timeoutIntervalForTimeAdmin\":" << tTimeAdmin << ","
					<< "\"timeoutIntervalForExport\":" << tExport 
				<< "},"
				<< mJsonFooter
			<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::GetTimeOutInterval(std::string& tAdmin, std::string& tTimeAdmin, std::string& tExport)
{
	std::string opfunction = "GetTimeOutInterval";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	tAdmin = GetValueFromJsonKey(jsonOutput, "timeoutIntervalForAdmin", AS_NONSTRING);
	tTimeAdmin = GetValueFromJsonKey(jsonOutput, "timeoutIntervalForTimeAdmin", AS_NONSTRING);
	tExport = GetValueFromJsonKey(jsonOutput, "timeoutIntervalForExport", AS_NONSTRING);

	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::SetUpForPrinter()
{
	std::string opfunction = "SetUpForPrinter";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::RunTSESelfTest()
{
	std::string opfunction = "RunTSESelfTest";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::DisableExportIfCspTestFails()
{
	std::string opfunction = "DisableExportIfCspTestFails";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

std::string TSEOperate::EnableExportIfCspTestFails()
{
	std::string opfunction = "EnableExportIfCspTestFails";
	std::string jsonResult = "";
	std::string jsonOutput = "";
	std::stringstream ssJson;

	ssJson << "{"
		<< mJsonHeader
		<< "\"function\" : \"" << opfunction << "\","
		<< "\"input\" : {"
		<< "},"
		<< mJsonFooter
		<< "}";
	mTseAcs->SendJsonStringToTse(ssJson.str(), jsonResult);

	if (GetValueFromJsonKey(jsonResult, "function", AS_STRING) != opfunction)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Invalid returned JSON string for operate function=%s", opfunction.c_str()));
		throw std::runtime_error("");
	}

	jsonOutput = GetJsonObj(jsonResult, "output");
	return GetValueFromJsonKey(jsonResult, "result", AS_STRING);
}

//Private functions

std::string TSEOperate::GetJsonObj(const std::string& tJsonStr, const std::string& tObj)
{
	std::string jStr = tJsonStr;

	if (mConMethod == ACCESS_BY_EPOS_DEVICE)
	{
		size_t index = 0;
		while (true)
		{
			index = jStr.find("&quot;", index);
			if (index == std::string::npos) break;

			jStr.replace(index, 6, "\"");
			index += 1;
		}
	}

	return TseSimpleJsonAndXmlParser::JsonGetObject(jStr, tObj);
}

std::string TSEOperate::GetValueFromJsonKey(const std::string& tJsonStr, const std::string& tKey, int tValType)
{
	std::string jStr = tJsonStr;

	if (mConMethod == ACCESS_BY_EPOS_DEVICE)
	{
		size_t index = 0;
		while (true)
		{
			index = jStr.find("&quot;", index);
			if (index == std::string::npos) break;

			jStr.replace(index, 6, "\"");
			index += 1;
		}

		index = 0;
		while (true)
		{
			index = jStr.find("&apos;", index);
			if (index == std::string::npos) break;

			jStr.replace(index, 6, "'");
			index += 1;
		}

		
	}
	if (tValType == AS_STRING)
	{
		return TseSimpleJsonAndXmlParser::JsonGetValue(jStr, tKey);
	}
	else if (tValType == AS_NONSTRING)
	{
		return TseSimpleJsonAndXmlParser::JsonGetNonStringValue(jStr, tKey);
	}
	else if (tValType == AS_ARRAY)
	{
		return TseSimpleJsonAndXmlParser::JsonGetArray(jStr, tKey);
	}

	return TseSimpleJsonAndXmlParser::JsonGetValue(jStr, tKey);
}

//EOF