// Copyright 2020-2021 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2021/05  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include <iostream>
#include <sstream>
#include <algorithm>
#include <ctime>
#include <iomanip> 
#include "../../Core/TSEOptions.h"
#include "../../Core/TseSimpleJsonAndXmlParser.h"
#include "../../Core/TSEUtil.h"
#include "../Include/EpsonTSELibStatusCodes.h"
#include "LoadEpsonTseDll.h"
#include "CuiLibCTester.h"

extern EpsonTse_GetTseDeviceList				E_GetTseDeviceList;
extern EpsonTse_TSEConnect						E_TSEConnect;
extern EpsonTse_TSEDisconnect					E_TSEDisconnect;
extern EpsonTse_TSEOpenDevice					E_TSEOpenDevice;
extern EpsonTse_TSECloseDevice					E_TSECloseDevice;
extern EpsonTse_Setup							E_Setup;
extern EpsonTse_RunTSESelfTest					E_RunTSESelfTest;
extern EpsonTse_FactoryReset					E_FactoryReset;
extern EpsonTse_GetRawStorageInfo				E_GetRawStorageInfo;
extern EpsonTse_GetStorageInfo					E_GetStorageInfo;
extern EpsonTse_GetRawStorageSmartInfo			E_GetRawStorageSmartInfo;
extern EpsonTse_GetStorageSmartInfo				E_GetStorageSmartInfo;
extern EpsonTse_GetStartedTransactionList		E_GetStartedTransactionList;
extern EpsonTse_GetLastTransactionResponse		E_GetLastTransactionResponse;
extern EpsonTse_GetMaxNumberOfClients			E_GetMaxNumberOfClients;
extern EpsonTse_GetCurrentNumberOfClients		E_GetCurrentNumberOfClients;
extern EpsonTse_GetMaxNumberOfTransactions		E_GetMaxNumberOfTransactions;
extern EpsonTse_GetCurrentNumberOfTransactions  E_GetCurrentNumberOfTransactions;
extern EpsonTse_ExportCertificate				E_ExportCertificate;
extern EpsonTse_ExportSerialNumber				E_ExportSerialNumber;
extern EpsonTse_AuthenticateUser				E_AuthenticateUser;
extern EpsonTse_LogOut							E_LogOut;
extern EpsonTse_ChangePin						E_ChangePin;
extern EpsonTse_UpdateTime						E_UpdateTime;
extern EpsonTse_DisableSecureElement			E_DisableSecureElement;
extern EpsonTse_RegisterTSEToHost				E_RegisterTSEToHost;
extern EpsonTse_RegisterSecretKey				E_RegisterSecretKey;
extern EpsonTse_RegisterClient					E_RegisterClient;
extern EpsonTse_DeregisterClient				E_DeregisterClient;
extern EpsonTse_LockTse							E_LockTse;
extern EpsonTse_UnlockTse						E_UnlockTse;
extern EpsonTse_EnableExportIfCspTestFails		E_EnableExportIfCspTestFails;
extern EpsonTse_DisableExportIfCspTestFails		E_DisableExportIfCspTestFails;
extern EpsonTse_SetTimeOutInterval				E_SetTimeOutInterval;
extern EpsonTse_GetTimeOutInterval				E_GetTimeOutInterval;
extern EpsonTse_GetAuthenticatedUserList		E_GetAuthenticatedUserList;
extern EpsonTse_GetRegisteredClientList			E_GetRegisteredClientList;
extern EpsonTse_StartTransaction				E_StartTransaction;
extern EpsonTse_UpdateTransaction				E_UpdateTransaction;
extern EpsonTse_FinishTransaction				E_FinishTransaction;
extern EpsonTse_ExportDataFilteredByTransactionNumberAndClientId			E_ExportDataFilteredByTransactionNumberAndClientId;
extern EpsonTse_ExportDataFilteredByTransactionNumber						E_ExportDataFilteredByTransactionNumber;
extern EpsonTse_ExportDataFilteredByTransactionNumberIntervalAndClientId	E_ExportDataFilteredByTransactionNumberIntervalAndClientId;
extern EpsonTse_ExportDataFilteredByTransactionNumberInterval				E_ExportDataFilteredByTransactionNumberInterval;
extern EpsonTse_ExportDataFilteredByPeriodOfTimeAndClientId					E_ExportDataFilteredByPeriodOfTimeAndClientId;
extern EpsonTse_ExportDataFilteredByPeriodOfTime							E_ExportDataFilteredByPeriodOfTime;
extern EpsonTse_ExportData						E_ExportData;
extern EpsonTse_FinalizeExport					E_FinalizeExport;
extern EpsonTse_CancelExport					E_CancelExport;
extern EpsonTse_ChangePuk						E_ChangePuk;
extern EpsonTse_UnblockUser						E_UnblockUser;
extern EpsonTse_TSEMemDealloc					E_TSEMemDealloc;
extern EpsonTse_TSEMemDeallocTseInfoObject		E_TSEMemDeallocTseInfoObject;
extern EpsonTse_TSESetLog						E_TSESetLog;
extern EpsonTse_TSEGetErrorDescription			E_TSEGetErrorDescription;


#define MENU_COLOR_HEADER  (FOREGROUND_RED  | FOREGROUND_INTENSITY)
#define MENU_COLOR_RESULT  (FOREGROUND_BLUE | FOREGROUND_INTENSITY)
#define MENU_COLOR_DEFAULT (FOREGROUND_RED  | FOREGROUND_GREEN | FOREGROUND_BLUE)
#define MENU_COLOR_WARNING (FOREGROUND_RED  | FOREGROUND_GREEN)

#define EPSON_TSE_ADMIN		"Administrator"
#define EXPORTFILEEXT		".tar"

CuiLibCTester::CuiLibCTester()
{
	mIsAdminLogin = false;
	mIsTseInitialized = false;
	mLastExportFile = "";
	mErrorDescLang = ERROR_DESC_EN;

	mConsole = GetStdHandle(STD_OUTPUT_HANDLE);
	if (mConsole == INVALID_HANDLE_VALUE)
	{
		return;
	}
}

CuiLibCTester::~CuiLibCTester()
{
}

bool CuiLibCTester::SetErrorDescLanguage(short int tLanguage)
{
	if (tLanguage == ERROR_DESC_DE)
	{
		mErrorDescLang = ERROR_DESC_DE;
	}
	else
	{
		mErrorDescLang = ERROR_DESC_EN;
	}
	return true;
}

bool CuiLibCTester::ConnectToTse(const TSEOptions& tTseOpt)
{
	short int status_code = EXECUTION_OK;

	status_code = E_TSEConnect(
		tTseOpt.mMethod,
		(unsigned char *)tTseOpt.mIdentifier.c_str(),
		(unsigned long) tTseOpt.mIdentifier.length(),
		(unsigned char *)tTseOpt.mTseHandler.c_str(),
		(unsigned long)tTseOpt.mTseHandler.length()
	);

	if (status_code != EXECUTION_OK)
	{
		return false;
	}

	return true;
}

bool CuiLibCTester::DisconnectFromTse()
{
	short int status_code = EXECUTION_OK;

	status_code = E_TSEDisconnect();

	if (status_code != EXECUTION_OK)
	{
		return false;
	}

	return true;
}

bool CuiLibCTester::CuiGetTseId(const std::string &tIpAddress, std::string& tTseId)
{
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned char *tse_list = NULL;
	unsigned long int tse_list_size = 0;
	std::string ip_address = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET TSE DEVICE LIST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	result = E_GetTseDeviceList(
		(unsigned char *)tIpAddress.c_str(),
		(unsigned long) tIpAddress.length(),
		&tse_list,
		&tse_list_size
	);

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;

	if ((result != EXECUTION_OK) && (result != OTHER_ERROR_CANNOT_GET_TSE_LIST))
	{
		return false;
	}

	if (result == OTHER_ERROR_CANNOT_GET_TSE_LIST)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "Cannot retrieve the TSE device list. Trying TSE=\"local_TSE\"..." << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
		
		tTseId = "local_TSE";
	}
	else
	{
		unsigned short tse_cnt = 0;
		std::string tseId = "";
		if (tse_list_size != 0)
		{
			std::cout << "List of TSE Devices : ";
			unsigned long ptr_index = 0;
			do
			{
				tseId = std::string((const char *)tse_list + ptr_index);
				std::cout << std::endl << "    " << tseId << ",";

				ptr_index += (unsigned long)(tseId.length() + 1);
				tse_cnt++;
			} while (ptr_index < tse_list_size);
			std::cout << std::endl;
			E_TSEMemDealloc(tse_list);
		}
		if ((tse_cnt == 1) && (tseId == "local_TSE"))
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "Trying TSE=\""<< tseId <<"\"..." << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

			tTseId = tseId;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			std::cout << "To exit, type bye. To connect to a TSE, enter the TSE ID: ";
			std::getline(std::cin, tTseId);
		}
	}

	return true;
}

bool CuiLibCTester::ShowMenu()
{	
	std::string sel_opt = "";
	int n_opt = -1;

	EPSON_TSE_INFO tseInfo;

	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return false;
	}

	result = E_GetStorageInfo(&tseInfo);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return false;
	}
	
	std::string initializeStatus = std::string((const char *)tseInfo.tseInitializationState);
	E_TSEMemDeallocTseInfoObject(&tseInfo);

	if ((result == EXECUTION_OK) || (result == OTHER_ERROR_CURRENTLY_EXPORTING) || (result == OTHER_ERROR_TSE_ALREADY_SET_UP))
	{
		if (initializeStatus == "INITIALIZED") 
		{
			mIsTseInitialized = true;
		}

		if ((result == OTHER_ERROR_CURRENTLY_EXPORTING) || (result == OTHER_ERROR_TSE_ALREADY_SET_UP))
		{
			mIsTseInitialized = true;
		}
	
		while (true)
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
			std::cout << std::endl << "[MAIN MENU]" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			std::cout << "TSE INITIALIZED? : " << (mIsTseInitialized ? "Yes" : "No") << std::endl;
			if (result != EXECUTION_OK)
			{
				std::cout << "Other Status : " << result << std::endl;
			}
			std::cout << std::endl;

			std::cout << "Please select the operation you want to perform" << std::endl;
			std::cout << "[1] Get Storage - TSE Information" << std::endl;
			if (mIsTseInitialized)
			{
				std::cout << "[2] Operate as Admin" << std::endl;
				std::cout << "[3] Operate as TimeAdmin" << std::endl;
				std::cout << "[4] Operate as Host" << std::endl;
				std::cout << "[5] Run TSE Self Test" << std::endl;
				std::cout << "[6] Get Log Message Certificate" << std::endl;
				std::cout << "[7] Export Serial Number" << std::endl;
				std::cout << "[8] Get Max Number of Clients" << std::endl;
				std::cout << "[9] Get Current Number of Clients" << std::endl;
				std::cout << "[10] Get Max Number of Transactions" << std::endl;
				std::cout << "[11] Get Current Number of Transactions" << std::endl;
				std::cout << "[12] Get Last Transaction Response" << std::endl;
				std::cout << "[13] Get Started Transaction List" << std::endl;
				std::cout << "[14] Get Storage - Smart Information" << std::endl;
			}
			else
			{
				std::cout << "[2] Set-up TSE" << std::endl;
				std::cout << "[5] Run TSE Self Test" << std::endl;
			}
		
			std::cout << "[98] Factory Reset (Only available on the development version of the TSE)" << std::endl;
			std::cout << "[99] Exit Menu" << std::endl;
			std::cout << std::endl << "Enter only the number: ";
			std::cin >> sel_opt;

			if (TSEUtil::IsNumber(sel_opt))
			{
				std::stringstream ssTmp(sel_opt);
				ssTmp >> n_opt;
			}
			else
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}

			if (n_opt == 99)
			{
				return false;
			}

			if (mIsTseInitialized)
			{
				switch (n_opt)
				{
					case 1: result = CuiGetStorageInfo(); break;
					case 2: result = CuiAdmin(); break;
					case 3: result = CuiTimeAdmin();  break;
					case 4: result = CuiHost(); break;
					case 5: result = CuiRunTseSelfTest(); break;
					case 6: result = CuiGetLogMsgCertificate(); break;
					case 7: result = CuiExportSerialNumber(); break;
					case 8: result = CuiGetMaxNumberOfClients(); break;
					case 9: result = CuiGetCurrentNumberOfClients(); break;
					case 10: result = CuiGetMaxNumberOfTrx(); break;
					case 11: result = CuiGetCurrentNumberOfTrx(); break;
					case 12: result = CuiGetLastTrxResponse();  break;
					case 13: result = CuiGetStartedTrxList(); break;
					case 14: result = CuiGetStorageSmartInfo(); break;
					case 98: result = CuiFactoryReset(); break;
					default:
					{
						SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
						std::cout << "INVALID INPUT!" << std::endl;
						SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
						continue;
					}
				}
			}
			else
			{
				switch (n_opt)
				{
					case 1: result = CuiGetStorageInfo(); break;
					case 2: result = CuiSetup(); break;
					case 5: result = CuiRunTseSelfTest(); break;
					case 98: result = CuiFactoryReset(); break;
					default:
					{
						SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
						std::cout << "INVALID INPUT!" << std::endl;
						SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
						continue;
					}
				}
			}
		}//while
	}//if (result == "EXECUTION_OK") 
	std::cout << result << std::endl;
	return false;
}

short int CuiLibCTester::CuiAdmin()
{
	short int result = EXECUTION_OK;
	std::string sel_opt = "";
	int n_opt = -1;

	mCurrentUserId = EPSON_TSE_ADMIN;

	while (true)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[OPERATE AS ADMIN]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		std::cout << "LOGGED-IN? : " << (mIsAdminLogin ? "Yes" : "No") << std::endl << std::endl;

		std::cout << "Please select the operation you want to perform" << std::endl;
		if (mIsAdminLogin)
		{
			std::cout << "[1] Log-out" << std::endl;
			std::cout << "[2] Update Time for first time" << std::endl;
			std::cout << "[3] Update Time" << std::endl;
			std::cout << "[4] Set Timeout Interval" << std::endl;
			std::cout << "[5] Get Timeout Interval" << std::endl;
			std::cout << "[6] Enable Export If Csp Test Fails" << std::endl;
			std::cout << "[7] Disable Export If Csp Test Fails" << std::endl;
			std::cout << "[8] Set Up For Printer (Register TSE to Host)" << std::endl;

			std::cout << "[10] Get Export Data (Archive Export)" << std::endl;
			std::cout << "[11] Get Export Data (Export Filtered By Transaction Number)" << std::endl;
			std::cout << "[12] Get Export Data (Export Filtered By Transaction Number Interval)" << std::endl;
			std::cout << "[13] Get Export Data (Export Filtered By Period Of Time)" << std::endl;
			std::cout << "[14] Finalize Export" << std::endl;
			std::cout << "[15] Cancel Export" << std::endl;

			std::cout << "[20] Register Shared Secret Key" << std::endl;
			std::cout << "[21] Register Client Id" << std::endl;
			std::cout << "[22] Deregister Client Id" << std::endl;
			std::cout << "[23] Unlock TSE" << std::endl;
			std::cout << "[24] Lock TSE" << std::endl;
			std::cout << "[25] Disable Secure Element" << std::endl;
			std::cout << "[26] Change Admin PIN" << std::endl;

			std::cout << "[30] Get Authenticated User List" << std::endl;
			std::cout << "[31] Get Registered Client Id List" << std::endl;
			std::cout << "[98] Update Time (Manual)" << std::endl;
		}
		else
		{
			std::cout << "[1] Log-in" << std::endl;
		}
		std::cout << "[99] Back" << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> sel_opt;
		
		if (TSEUtil::IsNumber(sel_opt))
		{
			std::stringstream ssTmp(sel_opt);
			ssTmp >> n_opt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}
		
		if (n_opt == 99)
		{
			break;
		}

		switch (n_opt)
		{
			case 1: 
			{
				if (mIsAdminLogin) 
					result = CuiAdminLogout();
				else 
					result = CuiAdminLogin();
				break;
			}
			case 2: result = CuiUpdateTimeFirstTime();  break;
			case 3: result = CuiUpdateTime();  break;
			case 4: result = CuiAdminSetTimeout(); break;
			case 5: result = CuiAdminGetTimeout(); break;
			case 6: result = CuiAdminEnableExportIfCspTestFails(); break;
			case 7: result = CuiAdminDisableExportIfCspTestFails(); break;
			case 8: result = CuiAdminSetUpForPrinter(); break;

			case 10: result = CuiAdminGetExportDataArchive(); break;
			case 11: result = CuiAdminGetExportDataFilterByTrx(); break;
			case 12: result = CuiAdminGetExportDataFilterByTrxNoInterval(); break;
			case 13: result = CuiAdminGetExportDataFilterByPeriod(); break;
			case 14: result = CuiAdminFinalizeExport(); break;
			case 15: result = CuiAdminCancelExportData(); break;

			case 20: result = CuiAdminRegisterSecretKey();  break;
			case 21: result = CuiAdminRegisterClient();  break;
			case 22: result = CuiAdminDeregisterClient();  break;
			case 23: result = CuiAdminUnlockTse();  break;
			case 24: result = CuiAdminLockTse();  break;
			case 25: result = CuiAdminDisableSecureElement();  break;
			case 26: result = CuiChangeUserPin(); break;

			case 30: result = CuiAdminGetAuthenticatedUserList();  break;
			case 31: result = CuiAdminGetRegisteredClientList();  break;
			case 98: result = CuiUpdateTimeManually();  break;
			case 99: break;
			default: 
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
		}
	}//while (true)
	return EXECUTION_OK;
}

short int CuiLibCTester::CuiGetStorageInfo()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];


	std::string y_or_n = "n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET STORAGE TSE INFO]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << std::endl << "Get TSE Info in raw JSON string format (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		unsigned char *tse_raw_info = NULL;
		unsigned long tse_raw_length = 0;

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_GetRawStorageInfo(
			&tse_raw_info,
			&tse_raw_length);

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		if (result == EXECUTION_OK)
		{
			std::cout << "TSE Info in raw JSON string:" << std::endl << std::string((const char *)tse_raw_info, tse_raw_length) << std::endl;
			E_TSEMemDealloc(tse_raw_info);
		}
	}
	else
	{
		EPSON_TSE_INFO tse_info;

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_GetStorageInfo(
			&tse_info);

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		if (result == EXECUTION_OK)
		{
			std::cout << "TSE Info:" << std::endl;
			std::cout << "\t cdcHash=" << tse_info.cdcHash << std::endl;
			std::cout << "\t cdcId=" << tse_info.cdcId << std::endl;
			std::cout << "\t certificateExpirationDate=" << tse_info.certificateExpirationDateStr << " (" << tse_info.certificateExpirationDate << ")" << std::endl;
			std::cout << "\t createdSignatures=" << tse_info.createdSignatures << std::endl;
			std::cout << "\t hardwareVersion=" << tse_info.hardwareVersion << std::endl;
			std::cout << "\t hasPassedSelfTest=" << ((tse_info.hasPassedSelfTest) ? "true" : "false") << std::endl;
			std::cout << "\t hasValidTime=" << ((tse_info.hasValidTime) ? "true" : "false") << std::endl;
			std::cout << "\t isTransactionInProgress=" << ((tse_info.isTransactionInProgress) ? "true" : "false") << std::endl;
			std::cout << "\t isExportEnabledIfCspTestFails=" << ((tse_info.isExportEnabledIfCspTestFails) ? "true" : "false") << std::endl;
			std::cout << "\t isTSEUnlocked=" << ((tse_info.isTSEUnlocked) ? "true" : "false") << std::endl;
			std::cout << "\t lastExportExecutedDate=" << tse_info.lastExportExecutedDateStr << " (" << tse_info.lastExportExecutedDate << ")" << std::endl;
			std::cout << "\t maxRegisteredClients=" << tse_info.maxRegisteredClients << std::endl;
			std::cout << "\t maxSignatures=" << tse_info.maxSignatures << std::endl;
			std::cout << "\t maxStartedTransactions=" << tse_info.maxStartedTransactions << std::endl;
			std::cout << "\t maxUpdateDelay=" << tse_info.maxUpdateDelay << std::endl;
			std::cout << "\t registeredClients=" << tse_info.registeredClients << std::endl;
			std::cout << "\t remainingSignatures=" << tse_info.remainingSignatures << std::endl;
			std::cout << "\t serialNumber=";
			for (unsigned long i = 0; i < tse_info.serialNumberLength; i++)
			{
				std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)tse_info.serialNumber[i] << " ";
			}
			std::cout << std::dec << std::endl;
			std::cout << "\t signatureAlgorithm=" << tse_info.signatureAlgorithm << std::endl;
			std::cout << "\t softwareVersion=" << tse_info.softwareVersion << std::endl;
			std::cout << "\t startedTransactions=" << tse_info.startedTransactions << std::endl;
			std::cout << "\t tarExportSize=" << tse_info.tarExportSize << std::endl;
			std::cout << "\t timeUntilNextSelfTest=" << tse_info.timeUntilNextSelfTest << std::endl;
			std::cout << "\t tseCapacity=" << tse_info.tseCapacity << std::endl;
			std::cout << "\t tseCurrentSize=" << tse_info.tseCurrentSize << std::endl;
			std::cout << "\t tseDescription=" << tse_info.tseDescription << std::endl;
			std::cout << "\t tseInitializationState=" << tse_info.tseInitializationState << std::endl;
			std::cout << "\t tsePublicKey=";
			for (unsigned long i = 0; i < tse_info.tsePublicKeyLength; i++)
			{
				std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)tse_info.tsePublicKey[i] << " ";
			}
			std::cout << std::dec << std::endl;
			std::cout << "\t vendorType=" << tse_info.vendorType << std::endl;
			std::cout << "\t rawInfo=\n" << std::string((const char *)tse_info.rawTseInfo, tse_info.rawTseInfoLength) << std::endl;

			E_TSEMemDeallocTseInfoObject(&tse_info);
		}
	}
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return result;
}

short int CuiLibCTester::CuiGetStorageSmartInfo()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];


	std::string y_or_n = "n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET STORAGE SMART INFO]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << std::endl << "Get Smart Info in raw JSON string format (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		unsigned char *smart_raw_info = NULL;
		unsigned long smart_raw_length = 0;

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_GetRawStorageSmartInfo(
			&smart_raw_info,
			&smart_raw_length);

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		if (result == EXECUTION_OK)
		{
			std::cout << "Smart Info in raw JSON string:" << std::endl << std::string((const char *)smart_raw_info, smart_raw_length) << std::endl;
			E_TSEMemDealloc(smart_raw_info);
		}
	}
	else
	{
		EPSON_TSE_SMART_INFO smart_info;

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_GetStorageSmartInfo(
			&smart_info);

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		if (result == EXECUTION_OK)
		{
			std::cout << "Smart Info:" << std::endl;
			std::cout << "\t dataIntegrity:" << std::endl;
			std::cout << "\t\t healthStatus=" << smart_info.dataIntegrity_healthStatus << std::endl;
			std::cout << "\t\t uncorrectableECCErrors=" << smart_info.dataIntegrity_uncorrectableECCErrors << std::endl;
			std::cout << "\t eraseLifetimeStatus:" << std::endl;
			std::cout << "\t\t healthStatus=" << smart_info.eraseLifetimeStatus_healthStatus << std::endl;
			std::cout << "\t\t remainingEraseCounts=" << smart_info.eraseLifetimeStatus_remainingEraseCounts << std::endl;
			std::cout << "\t spareBlockStatus:" << std::endl;
			std::cout << "\t\t healthStatus=" << smart_info.spareBlockStatus_healthStatus << std::endl;
			std::cout << "\t\t remainingSpareBlocks=" << smart_info.spareBlockStatus_remainingSpareBlocks << std::endl;

			std::cout << "\t remainingTenYearsDataRetention=" << smart_info.remainingTenYearsDataRetention << std::endl;
			std::cout << "\t isReplacementNeeded=" << ((smart_info.isReplacementNeeded) ? "true" : "false") << std::endl;
			std::cout << "\t tseHealth=" << smart_info.tseHealth << std::endl;
		}
	}
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return result;
}

short int CuiLibCTester::CuiRunTseSelfTest()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[RUN TSE SELF TEST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_RunTSESelfTest();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	return result;
}

short int CuiLibCTester::CuiSetup()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string puk = "";
	std::string admin_pin = "";
	std::string time_admin_pin = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[SET UP]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "PUK: ";
	std::getline(std::cin, puk);

	std::cout << "Admin PIN: ";
	std::getline(std::cin, admin_pin);

	std::cout << "Time Admin PIN: ";
	std::getline(std::cin, time_admin_pin);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_Setup(
		(unsigned char *) admin_pin.c_str(),
		(unsigned long) admin_pin.length(),
		(unsigned char *) time_admin_pin.c_str(),
		(unsigned long) time_admin_pin.length(),
		(unsigned char *) puk.c_str(),
		(unsigned long) puk.length()
		);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == EXECUTION_OK)
	{
		mIsTseInitialized = true;
	}

	return result;
}

short int CuiLibCTester::CuiFactoryReset()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[FACTORY RESET]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "WARNING!" << std::endl;
	std::cout << "Please note that this function exists only for development purposes and will not be available for the final TSEs." << std::endl;

	std::cout << std::endl << "Are you sure you want to do factory reset (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_FactoryReset();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == EXECUTION_OK)
		{
			mIsTseInitialized = false;
		}
	}

	return result;
}

short int CuiLibCTester::CuiAdminLogin()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string pin = "";
	std::string secret_key = "";
	std::string admin = EPSON_TSE_ADMIN;
	short int remaining_retries = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-IN AS ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "PIN: ";
	std::getline(std::cin, pin);
	std::cout << "Secret Key: ";
	std::getline(std::cin, secret_key);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_AuthenticateUser(
		(unsigned char *)admin.c_str(),
		(unsigned long) admin.length(),
		(unsigned char *)pin.c_str(),
		(unsigned long) pin.length(),
		(unsigned char *)secret_key.c_str(),
		(unsigned long)secret_key.length(),
		&remaining_retries
		);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang,result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Remaining retries: " << remaining_retries << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == EXECUTION_OK)
	{
		mIsAdminLogin = true;
	}

	return result;
}

short int CuiLibCTester::CuiAdminLogout()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";
	std::string admin = EPSON_TSE_ADMIN;

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-OUT ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to log-out (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_LogOut(
			(unsigned char *)admin.c_str(),
			(unsigned long) admin.length()
		);

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if ((result == EXECUTION_OK) || (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER))
		{
			mIsAdminLogin = false;
		}
	}

	return result;
}

short int CuiLibCTester::CuiAdminSetTimeout()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned long timeout_admin = 0;
	unsigned long timeout_timeadmin = 0;
	unsigned long timeout_export = 0;

	std::string timeout_admin_s = "";
	std::string timeout_timeadmin_s = "";
	std::string timeout_export_s = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[SET TIMEOUT INTERVAL]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Note: Defaut timeouts are: 900 sec for Admin, 28800 sec for Time admin, 100 sec for Export." << std::endl;

	std::cout << "Please enter the following (in seconds):" << std::endl;
	getchar();
	while (true)
	{
		std::cout << "New Timeout for Admin: ";
		std::getline(std::cin, timeout_admin_s);
		if (TSEUtil::IsNumber(timeout_admin_s))
		{
			std::stringstream ss;
			ss << timeout_admin_s;
			ss >> timeout_admin;
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "New Timeout for TimeAdmin: ";
		std::getline(std::cin, timeout_timeadmin_s);
		if (TSEUtil::IsNumber(timeout_timeadmin_s))
		{
			std::stringstream ss;
			ss << timeout_timeadmin_s;
			ss >> timeout_timeadmin;
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "New Timeout for Export: ";
		std::getline(std::cin, timeout_export_s);
		if (TSEUtil::IsNumber(timeout_export_s))
		{
			std::stringstream ss;
			ss << timeout_export_s;
			ss >> timeout_export;
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_SetTimeOutInterval(
		timeout_admin,
		timeout_timeadmin,
		timeout_export);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Timeout Admin: " << timeout_admin << std::endl;
	std::cout << "Timeout Timeadmin: " << timeout_timeadmin << std::endl;
	std::cout << "Timeout Export: " << timeout_export << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminGetTimeout()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned long timeout_admin = 0;
	unsigned long timeout_timeadmin = 0;
	unsigned long timeout_export = 0;

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET TIMEOUT INTERVAL]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Note: Defaut timeouts are: 900 sec for Admin, 28800 sec for Time admin, 100 sec for Export." << std::endl;

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetTimeOutInterval(
				&timeout_admin,
				&timeout_timeadmin,
				&timeout_export);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Timeout Admin: " << timeout_admin << std::endl;
	std::cout << "Timeout Timeadmin: " << timeout_timeadmin << std::endl;
	std::cout << "Timeout Export: " << timeout_export << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminSetUpForPrinter()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[SET UP FOR PRINTER (REGISTER TSE TO HOST)]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_RegisterTSEToHost();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}
	return result;
}

short int CuiLibCTester::CuiUpdateTimeFirstTime() 
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TIME FOR FIRST TIME]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "This function is ignored if Data/Time has already been set-up." << std::endl;

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	
	time_t rawtime = std::time(nullptr);
	result = E_UpdateTime(
				(unsigned char *)mCurrentUserId.c_str(),
				(unsigned long) mCurrentUserId.length(),
				(unsigned long int) rawtime,
				true);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER) || (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}

	return result;
}

short int CuiLibCTester::CuiUpdateTime()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TIME]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	time_t timenow = std::time(nullptr);
	result = E_UpdateTime(
		(unsigned char *)mCurrentUserId.c_str(),
		(unsigned long) mCurrentUserId.length(),
		(unsigned long int) timenow,
		false);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER) || (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}

	return result;
}

short int CuiLibCTester::CuiUpdateTimeManually()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string new_date = "";
	std::string new_time = "";
	std::string new_datetime_s = "";
	unsigned long int new_datetime = 0;
	
	std::string y_or_n = "";
	bool update_for_first = false;
	
	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TIME (Manual)]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	
	std::cout << std::endl << "UpdateTime for the first time? (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		update_for_first = true;
	}

	getchar();
	while (true)
	{
		std::cout << "Enter the new Date (Format YYYY-MM-DD): ";
		std::getline(std::cin, new_date);
		std::string tmpDate = new_date + "T00:00:00Z";
		if (TSEUtil::IsDateTime(tmpDate))
		{
			break;
		}

		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	while (true)
	{
		std::cout << "Enter the new Time (Format HH:MM:SS): ";
		std::getline(std::cin, new_time);
		std::string tmpTime = "2020-01-01T" + new_time + "Z";
		if (TSEUtil::IsDateTime(tmpTime))
		{
			break;
		}

		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	new_datetime_s = new_date + "T" + new_time + "Z";

	if (TSEUtil::IsDateTime(new_datetime_s))
	{
		std::stringstream ss;
		struct tm stm;

		ss << new_datetime_s;
		ss >> std::get_time(&stm, "%Y-%m-%dT%H:%M:%SZ");
		new_datetime = (unsigned long int) _mkgmtime(&stm);
	}
	else
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
		return OTHER_ERROR;;
	}
	
	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_UpdateTime(
		(unsigned char *)mCurrentUserId.c_str(),
		(unsigned long)mCurrentUserId.length(),
		(unsigned long int) new_datetime,
		update_for_first);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER) || (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}
	return result;
}


short int CuiLibCTester::CuiAdminEnableExportIfCspTestFails()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[ENABLE EXPORT IF CSP TEST FAILS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_EnableExportIfCspTestFails();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}
	return result;
}

short int CuiLibCTester::CuiAdminDisableExportIfCspTestFails()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[DISABLE EXPORT IF CSP TEST FAILS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_DisableExportIfCspTestFails();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}


		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}
	return result;
}


short int CuiLibCTester::CuiAdminGetExportDataArchive()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned char * export_data = NULL;
	unsigned long export_data_length = 0;

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - ARCHIVE EXPORT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_ExportData(
		&export_data,
		&export_data_length
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == EXECUTION_OK)
	{
		std::string export_filename = "";
		std::string temp_date = TSEUtil::GetCurrentUtcTime();
		temp_date.erase(std::remove(temp_date.begin(), temp_date.end(), ':'), temp_date.end());
		export_filename = "Export_ARCHIVE_" + temp_date + std::string(EXPORTFILEEXT);
		remove(export_filename.c_str());

		TSEUtil::SaveToFile(export_filename, export_data, export_data_length);
		mLastExportFile = export_filename;
		std::cout << "Export File Size=\"" << export_data_length << "\"" << std::endl;
		std::cout << "Export File: " << export_filename << std::endl;
	
		E_TSEMemDealloc(export_data);
	}

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminGetExportDataFilterByTrx() 
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string client_id;
	std::string trx_no_s;

	unsigned long trx_no = 0;
	unsigned char *export_data = NULL;
	unsigned long export_data_length = 0;

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - EXPORT FILTERED BY TRANSACTION NUMBER]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	
	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, client_id);

	while (true)
	{
		std::cout << "Enter the Transaction No: ";
		std::getline(std::cin, trx_no_s);
		if (TSEUtil::IsNumber(trx_no_s))
		{
			std::stringstream ss;
			ss << trx_no_s;
			ss >> trx_no;
			break;
		}
			
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	if (client_id == "")
	{
		result = E_ExportDataFilteredByTransactionNumber(
			trx_no,
			&export_data,
			&export_data_length
		);
	}
	else
	{
		result = E_ExportDataFilteredByTransactionNumberAndClientId(
			trx_no,
			(unsigned char *)client_id.c_str(),
			(unsigned long) client_id.length(),
			&export_data,
			&export_data_length
		);
	}

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == EXECUTION_OK)
	{
		std::string export_filename = "";

		export_filename = "Export_" + client_id + "_TRXNUM_" + trx_no_s + EXPORTFILEEXT;
		remove(export_filename.c_str());

		TSEUtil::SaveToFile(export_filename, export_data, export_data_length);
		mLastExportFile = export_filename;
		std::cout << "Export File Size=\"" << export_data_length << "\"" << std::endl;
		std::cout << "Export File: " << export_filename << std::endl;

		E_TSEMemDealloc(export_data);
	}

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminGetExportDataFilterByTrxNoInterval()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string client_id;
	std::string start_trx_no_s, end_trx_no_s;

	unsigned long start_trx_no = 0, end_trx_no = 0;
	unsigned char *export_data = NULL;
	unsigned long export_data_length = 0;

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - EXPORT FILTERED BY TRANSACTION NUMBER INTERVAL]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, client_id);

	while (true)
	{
		std::cout << "Enter the Start Transaction No: ";
		std::getline(std::cin, start_trx_no_s);
		if (TSEUtil::IsNumber(start_trx_no_s))
		{
			std::stringstream ss;
			ss << start_trx_no_s;
			ss >> start_trx_no;
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "Enter the End Transaction No: ";
		std::getline(std::cin, end_trx_no_s);
		if (TSEUtil::IsNumber(end_trx_no_s))
		{
			std::stringstream ss;
			ss << end_trx_no_s;
			ss >> end_trx_no;
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	if (client_id == "")
	{
		result = E_ExportDataFilteredByTransactionNumberInterval(
			start_trx_no,
			end_trx_no,
			&export_data,
			&export_data_length
		);
	}
	else
	{
		result = E_ExportDataFilteredByTransactionNumberIntervalAndClientId(
			start_trx_no,
			end_trx_no,
			(unsigned char *)client_id.c_str(),
			(unsigned long) client_id.length(),
			&export_data,
			&export_data_length
		);
	}

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == EXECUTION_OK)
	{
		std::string export_filename = "";

		export_filename = "Export_" + client_id + "_TRXRNG_" + start_trx_no_s + "_TO_" + end_trx_no_s + EXPORTFILEEXT;
		remove(export_filename.c_str());

		TSEUtil::SaveToFile(export_filename, export_data, export_data_length);
		mLastExportFile = export_filename;
		std::cout << "Export File Size=\"" << export_data_length << "\"" << std::endl;
		std::cout << "Export File: " << export_filename << std::endl;

		E_TSEMemDealloc(export_data);
	}

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminGetExportDataFilterByPeriod()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string client_id;
	std::string start_date_s, end_date_s;

	unsigned long int start_date = 0, end_date = 0;
	unsigned char *export_data = NULL;
	unsigned long export_data_length = 0;

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - EXPORT FILTERED BY PERIOD OF TIME]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, client_id);
	
	while (true)
	{
		std::cout << "Enter the Start Date/Time (UTC Format YYYY-MM-DDTHH:MM:SSZ): ";
		std::getline(std::cin, start_date_s);
		if (TSEUtil::IsDateTime(start_date_s))
		{
			std::stringstream ss;
			struct tm stm;

			ss << start_date_s;
			ss >> std::get_time(&stm, "%Y-%m-%dT%H:%M:%SZ");
			start_date = (unsigned long int) _mkgmtime(&stm);
			break;
		}
			
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "Enter the End Date/Time (UTC Format YYYY-MM-DDTHH:MM:SSZ): ";
		std::getline(std::cin, end_date_s);
		if (TSEUtil::IsDateTime(end_date_s))
		{
			std::stringstream ss;
			struct tm stm;

			ss << end_date_s;
			ss >> std::get_time(&stm, "%Y-%m-%dT%H:%M:%SZ");
			end_date = (unsigned long int) _mkgmtime(&stm);
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	if (client_id == "")
	{
		result = E_ExportDataFilteredByPeriodOfTime(
			start_date,
			end_date,
			&export_data,
			&export_data_length
		);
	}
	else
	{
		result = E_ExportDataFilteredByPeriodOfTimeAndClientId(
			start_date,
			end_date,
			(unsigned char *)client_id.c_str(),
			(unsigned long) client_id.length(),
			&export_data,
			&export_data_length
		);
	}

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == EXECUTION_OK)
	{
		std::string export_filename = "";

		std::string tmp_sdate = start_date_s;
		std::string tmp_edate = end_date_s;
		tmp_sdate.erase(std::remove(tmp_sdate.begin(), tmp_sdate.end(), ':'), tmp_sdate.end());
		tmp_edate.erase(std::remove(tmp_edate.begin(), tmp_edate.end(), ':'), tmp_edate.end());
		
		export_filename = "Export_" + client_id + "_PERIOD_" + tmp_sdate + "_TO_" + tmp_edate + EXPORTFILEEXT;
		remove(export_filename.c_str());
		

		TSEUtil::SaveToFile(export_filename, export_data, export_data_length);
		mLastExportFile = export_filename;
		std::cout << "Export File Size=\"" << export_data_length << "\"" << std::endl;
		std::cout << "Export File: " << export_filename << std::endl;

		E_TSEMemDealloc(export_data);
	}

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminFinalizeExport()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";
	bool deletestoreddata = false;
	bool continuefinalize = false;
	
	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[FINALIZE EXPORT]" << std::endl;
	std::cout << std::endl << "Note: Execute this only after a successful and complete GetExport" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;
	if (y_or_n == "y")
	{
		y_or_n = "";
		std::cout << std::endl << "Do you want to delete the stored data (y/n): ";
		std::cin >> y_or_n;

		deletestoreddata = (y_or_n == "y") ? true : false;

		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_FinalizeExport(deletestoreddata);

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}

	return result;
}


short int CuiLibCTester::CuiAdminCancelExportData()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[CANCEL EXPORT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_CancelExport();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (mLastExportFile != "")
		{
			remove(mLastExportFile.c_str());
		}

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}
	return result;
}

short int CuiLibCTester::CuiAdminRegisterSecretKey()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string secret_key = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << "[REGISTER SHARED SECRET KEY]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "Secret Key: ";
	std::getline(std::cin, secret_key);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_RegisterSecretKey(
		(unsigned char *)secret_key.c_str(),
		(unsigned long) secret_key.length());

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminDisableSecureElement()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[DISABLE SECURE ELEMENT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_DisableSecureElement();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}
	return result;
}


short int CuiLibCTester::CuiChangeUserPin()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string pin = "";
	std::string new_pin = "";
	short int remaining_retries = 0;

	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return OTHER_ERROR;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[CHANGE PIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "Old PIN: ";
	std::getline(std::cin, pin);
	std::cout << "New PIN: ";
	std::getline(std::cin, new_pin);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_ChangePin(
		(unsigned char *)mCurrentUserId.c_str(),
		(unsigned long) mCurrentUserId.length(),
		(unsigned char *)pin.c_str(),
		(unsigned long) pin.length(),
		(unsigned char *)new_pin.c_str(),
		(unsigned long) new_pin.length(),
		&remaining_retries);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Remaining retries: " << remaining_retries << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER) || (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}

	return result;
}

short int CuiLibCTester::CuiAdminRegisterClient() 
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string client_id = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[REGISTER CLIENT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "New Client Id: ";
	std::getline(std::cin, client_id);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_RegisterClient(
				(unsigned char *)client_id.c_str(),
				(unsigned long) client_id.length());

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}
	
	return result;
}

short int CuiLibCTester::CuiAdminDeregisterClient()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string client_id = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[DEREGISTER CLIENT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please enter the client id to remove:" << std::endl;
	getchar();
	std::cout << "Client Id: ";
	std::getline(std::cin, client_id);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_DeregisterClient(
		(unsigned char *)client_id.c_str(),
		(unsigned long) client_id.length());

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return result;
}

short int CuiLibCTester::CuiAdminUnlockTse()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UNLOCK TSE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_UnlockTse();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}
	return result;
}

short int CuiLibCTester::CuiAdminLockTse()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOCK TSE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Are you sure you want to continue? (y/n) : ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_LockTse();

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
		{
			mIsAdminLogin = false;
		}
	}
	return result;
}

short int CuiLibCTester::CuiAdminGetAuthenticatedUserList() 
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string sel_opt = "";
	int n_opt = -1;
	std::string user_role = "";

	unsigned char *authenticated_user_list = NULL;
	unsigned long int authenticated_user_list_length = 0;
	while (true)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[GET AUTHENTICATED USER LIST]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		std::cout << "Select the number that correspond to the User Role you want to check:" << std::endl;
		std::cout << "[1] Admin " << std::endl;
		std::cout << "[2] TimeAdmin " << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> sel_opt;

		if (TSEUtil::IsNumber(sel_opt))
		{
			std::stringstream ssTmp(sel_opt);
			ssTmp >> n_opt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}

		if (n_opt == 1)
		{
			user_role = "Admin";
			break;
		}
		else if (n_opt == 2)
		{
			user_role = "TimeAdmin";
			break;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}
	}

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetAuthenticatedUserList(
		(unsigned char *) user_role.c_str(),
		(unsigned long) user_role.length(),
		&authenticated_user_list,
		&authenticated_user_list_length
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	if (authenticated_user_list_length != 0)
	{
		std::cout << "List of Authenticated User : ";
		unsigned long ptr_index = 0;
		do
		{
			std::string client_id = std::string((const char *)authenticated_user_list + ptr_index);
			std::cout << std::endl << "    " << client_id << ",";

			ptr_index += (unsigned long) (client_id.length() + 1);
		} while (ptr_index < authenticated_user_list_length);
		std::cout << std::endl;
		E_TSEMemDealloc(authenticated_user_list);
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return EXECUTION_OK;
}

short int CuiLibCTester::CuiAdminGetRegisteredClientList()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned char *registered_client_list = NULL;
	unsigned long int registered_client_list_length = 0;

	if (IsAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET REGISTERED CLIENT ID LIST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetRegisteredClientList(
		&registered_client_list,
		&registered_client_list_length
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	if (registered_client_list_length != 0)
	{
		std::cout << "List of Registered Client : ";
		unsigned long ptr_index = 0;
		do
		{
			std::string clientId = std::string((const char *)registered_client_list + ptr_index);
			std::cout << std::endl << "    " << clientId << ",";

			ptr_index += (unsigned long)(clientId.length() + 1);
		} while (ptr_index < registered_client_list_length);
		std::cout << std::endl;
		E_TSEMemDealloc(registered_client_list);
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER)
	{
		mIsAdminLogin = false;
	}

	return EXECUTION_OK;
}

short int CuiLibCTester::CuiGetLogMsgCertificate()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned char *certificate = NULL;
	unsigned long certificate_length = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET LOG MESSAGE CERTIFICATE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_ExportCertificate(
		&certificate,
		&certificate_length
	);
	
	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Log Certificate:" << std::endl << std::string((const char *)certificate, certificate_length) << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	E_TSEMemDealloc(certificate);

	return result;
}

short int CuiLibCTester::CuiGetLastTrxResponse()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string client_id = "";

	unsigned long log_time = 0;
	unsigned long trx_no = 0;
	unsigned long signature_cntr = 0;
	unsigned char* serial_no = NULL;
	unsigned char* signature = NULL;
	unsigned long serial_no_length = 0;
	unsigned long signature_length = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET LAST TRANSACTION RESPONSE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, client_id);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetLastTransactionResponse(
				(unsigned char *)client_id.c_str(),
				(unsigned long) client_id.length(),
				&trx_no,
				&log_time,
				&signature_cntr,
				&signature,
				&signature_length,
				&serial_no,
				&serial_no_length );

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;

	if (result == EXECUTION_OK)
	{
		struct tm stm;
		char log_time_s[sizeof("2019-01-01T23:59:59Z")] = "";
		time_t timet = log_time;

		if ((gmtime_s(&stm, &timet)) == 0)
		{
			strftime(log_time_s, sizeof(log_time_s), "%FT%TZ", &stm);
		}

		std::cout << "Transaction No : " << trx_no << std::endl;
		std::cout << "Log Time : " << log_time << " (" << log_time_s << ")" << std::endl;
		
		std::cout << "Serial Number : ";
		for (unsigned long i = 0; i < serial_no_length; i++)
		{
			std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)serial_no[i] << " ";
		}
		std::cout << std::dec << std::endl;

		std::cout << "Signature Counter : " << signature_cntr << std::endl;

		std::cout << "Signature Value : ";
		for (unsigned long i = 0; i < signature_length; i++)
		{
			std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)signature[i] << " ";
		}
		std::cout << std::dec << std::endl;

		E_TSEMemDealloc(serial_no);
		E_TSEMemDealloc(signature);
	}
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return result;
}


short int CuiLibCTester::CuiGetStartedTrxList()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned long int *transaction_list = NULL;
	unsigned long int transaction_list_size = 0;
	std::string client_id = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET STARTED TRANSACTION LIST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, client_id);


	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetStartedTransactionList(
		(unsigned char *) client_id.c_str(),
		(unsigned long) client_id.length(),
		&transaction_list,
		&transaction_list_size
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;

	std::cout << "List of Started transactions : ";
	for (unsigned long i = 0; i < transaction_list_size; i++)
	{
		std::cout << std::endl << "    " << (int)transaction_list[i] << ", ";
	}
	std::cout << std::dec << std::endl;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	E_TSEMemDealloc(transaction_list);

	return EXECUTION_OK;

}

short int CuiLibCTester::CuiExportSerialNumber()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned char *serial_number = NULL;
	unsigned long int serial_number_size = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET SERIAL NUMBER]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);


	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_ExportSerialNumber(
		&serial_number,
		&serial_number_size
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Serial Number : " ;
	for (unsigned long i = 0; i < serial_number_size; i++)
	{
		std::cout << std::setfill('0') << std::setw(2)  << std::hex << (int) serial_number[i] << " ";
	}
	std::cout << std::dec << std::endl;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	E_TSEMemDealloc(serial_number);
	

	return EXECUTION_OK;

}

short int CuiLibCTester::CuiGetMaxNumberOfClients()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned long int max_number_client = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET MAX NUMBER OF CLIENTS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetMaxNumberOfClients(
		&max_number_client
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Max no. of clients : " << max_number_client << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return EXECUTION_OK;
}

short int CuiLibCTester::CuiGetCurrentNumberOfClients()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned long int current_number_client = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET CURRENT NUMBER OF CLIENTS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);


	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetCurrentNumberOfClients(
		&current_number_client
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Current no. of clients : " << current_number_client << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return EXECUTION_OK;
}


short int CuiLibCTester::CuiGetMaxNumberOfTrx()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned long int max_number_trx = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET MAX NUMBER OF TRANSACTIONS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);


	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetMaxNumberOfTransactions(
		&max_number_trx
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Max no. of transactions : " << max_number_trx << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return EXECUTION_OK;
}

short int CuiLibCTester::CuiGetCurrentNumberOfTrx()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	unsigned long int current_number_trx = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET CURRENT NUMBER OF TRANSACTIONS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);


	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_GetCurrentNumberOfTransactions(
		&current_number_trx
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Current no. of transactions : " << current_number_trx << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return EXECUTION_OK;
}

short int CuiLibCTester::CuiTimeAdmin()
{
	short int result = EXECUTION_OK;
	std::string sel_opt = "";
	int n_opt = -1;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[OPERATE AS TIME ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please enter the Client Id." << std::endl;
	getchar();
	std::cout << "Client Id: ";
	std::getline(std::cin, mCurrentUserId);

	while (true)
	{
		bool isTimeAdminLogin = IsTimeAdminLogin();
		
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[OPERATE AS TIME ADMIN]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
		
		std::cout << "Client ID: " << mCurrentUserId << std::endl;
		std::cout << "LOGGED-IN? : " << (isTimeAdminLogin ? "Yes" : "No") << std::endl << std::endl;
		
		std::cout << "Please select the operation you want to perform" << std::endl;

		if (isTimeAdminLogin)
		{
			std::cout << "[1] Log-out" << std::endl;
			std::cout << "[2] Update Date/Time for the first time" << std::endl;
			std::cout << "[3] Update Date/Time" << std::endl;
			std::cout << "[4] Start Transaction" << std::endl;
			std::cout << "[5] Update Transaction" << std::endl;
			std::cout << "[6] Finish Transaction" << std::endl;
			std::cout << "[7] Change PIN For TimeAdmin" << std::endl;
		}
		else
		{
			std::cout << "[1] Log-in" << std::endl;
		}
		std::cout << "[99] Back" << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> sel_opt;
			
		if (TSEUtil::IsNumber(sel_opt))
		{
			std::stringstream ssTmp(sel_opt);
			ssTmp >> n_opt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}
			
		if (n_opt == 99)
		{
			break;
		}
		
		switch (n_opt)
		{
			case 1:
			{
				if (isTimeAdminLogin) 
				{
					result = CuiTimeAdminLogout();
				}
				else
				{ 
					result = CuiTimeAdminLogin();
				}
				break;
			}
			case 2: result = CuiUpdateTimeFirstTime();  break;
			case 3: result = CuiUpdateTime();  break;
			case 4: result = CuiTimeAdminStartTransaction(); break;
			case 5: result = CuiTimeAdminUpdateTransaction(); break;
			case 6: result = CuiTimeAdminFinishTransaction(); break;
			case 7: result = CuiChangeUserPin();  break;
			case 99: break;
			default:
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
		}
		
	}//while (true)
	return EXECUTION_OK;
}

short int CuiLibCTester::CuiTimeAdminLogin()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string pin = "";
	std::string secret_key = "";
	short int remaining_retries = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-IN AS TIME ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "PIN: ";
	std::getline(std::cin, pin);
	std::cout << "Secret Key: ";
	std::getline(std::cin, secret_key);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_AuthenticateUser(
		(unsigned char *)mCurrentUserId.c_str(),
		(unsigned long) mCurrentUserId.length(),
		(unsigned char *)pin.c_str(),
		(unsigned long) pin.length(),
		(unsigned char *)secret_key.c_str(),
		(unsigned long) secret_key.length(),
		&remaining_retries
	);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Remaining retries: " << remaining_retries << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == EXECUTION_OK)
	{
		mClientIdLogInList.push_back(mCurrentUserId);
	}

	return result;
}

short int CuiLibCTester::CuiTimeAdminLogout()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string y_or_n = "";

	if (IsTimeAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-OUT TIME ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to log-out (y/n): ";
	std::cin >> y_or_n;

	if (y_or_n == "y")
	{
		if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		result = E_LogOut(
			(unsigned char *)mCurrentUserId.c_str(),
			(unsigned long) mCurrentUserId.length()
		);

		if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
		{
			return status_code;
		}

		E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl;
		std::cout << "RESULT : " << result << std::endl;
		std::cout << "RESULT Description : " << result_desc << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if ((result == EXECUTION_OK) || (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER))
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}

	return result;
}


short int CuiLibCTester::CuiTimeAdminStartTransaction()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string process_type = "";
	std::string process_data = "";
	std::string addtnl_data = "";

	unsigned long log_time = 0;
	unsigned char* serial_no = NULL;
	unsigned char* signature = NULL;
	unsigned long serial_no_length = 0;
	unsigned long signature_length = 0;
	unsigned long trx_no = 0;
	unsigned long signature_cntr = 0;

	unsigned char* process_data_ptr = NULL;
	unsigned long process_data_len = 0;
	unsigned char* additnl_data_ptr = NULL;
	unsigned long additnl_data_len = 0;
	std::vector<unsigned char> process_data_bin;
	std::vector<unsigned char> additnl_data_bin;

	if (IsTimeAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[START TRANSACTION]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Type: ";
	std::getline(std::cin, process_type);

	while (true)
	{
		std::cout << "Process Data: ";
		std::getline(std::cin, process_data);
		if ((process_data.length() >= 3) && (process_data.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(process_data.substr(3), process_data_bin);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				process_data_ptr = process_data_bin.data();
				process_data_len = (unsigned long) process_data_bin.size();
				break;
			}			
		}//if
		else
		{
			process_data_ptr = (unsigned char *) process_data.c_str();
			process_data_len = (unsigned long) process_data.length();
			break;
		}
	}//while

	while (true)
	{
		std::cout << "Additional Data: ";
		std::getline(std::cin, addtnl_data);
		if ((addtnl_data.length() >= 3) && (addtnl_data.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(addtnl_data.substr(3), additnl_data_bin);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				additnl_data_ptr = additnl_data_bin.data();
				additnl_data_len = (unsigned long) additnl_data_bin.size();
				break;
			}
		}//if
		else
		{
			additnl_data_ptr = (unsigned char *)addtnl_data.c_str();
			additnl_data_len = (unsigned long) addtnl_data.length();
			break;
		}
	}//while

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_StartTransaction(
		(unsigned char *) mCurrentUserId.c_str(),
		(unsigned long) mCurrentUserId.length(),
		process_data_ptr,
		process_data_len,
		(unsigned char *)process_type.c_str(),
		(unsigned long) process_type.length(),
		additnl_data_ptr,
		additnl_data_len,
		&trx_no,
		&log_time,
		&serial_no,
		&serial_no_length,
		&signature_cntr,
		&signature,
		&signature_length);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	if (result == EXECUTION_OK)
	{
		struct tm stm;
		char log_time_s[sizeof("2019-01-01T23:59:59Z")] = "";
		time_t timet = log_time;

		if ((gmtime_s(&stm, &timet)) == 0)
		{
			strftime(log_time_s, sizeof(log_time_s), "%FT%TZ", &stm);
		}

		std::cout << "Transaction No : " << trx_no << std::endl;
		std::cout << "Log Time : " << log_time << " (" << log_time_s << ")" << std::endl;

		std::cout << "Serial Number : ";
		for (unsigned long i = 0; i < serial_no_length; i++)
		{
			std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)serial_no[i] << " ";
		}
		std::cout << std::dec << std::endl;

		std::cout << "Signature Counter : " << signature_cntr << std::endl;

		std::cout << "Signature Value : ";
		for (unsigned long i = 0; i < signature_length; i++)
		{
			std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)signature[i] << " ";
		}
		std::cout << std::dec << std::endl;

		E_TSEMemDealloc(serial_no);
		E_TSEMemDealloc(signature);
	}
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER)
	{
		mClientIdLogInList.remove(mCurrentUserId);
	}

	return result;
}

short int CuiLibCTester::CuiTimeAdminUpdateTransaction()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string process_type = "";
	std::string process_data = "";
	std::string trx_no_s = "";
	unsigned long trx_no = 0;

	unsigned long log_time = 0;
	unsigned char* signature = NULL;
	unsigned long signature_length = 0;
	unsigned long signature_cntr = 0;

	unsigned char* process_data_ptr = NULL;
	unsigned long process_data_len = 0;
	std::vector<unsigned char> process_data_bin;

	if (IsTimeAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TRANSACTION]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	getchar();
	while (true)
	{
		std::cout << "Transaction No: ";
		std::getline(std::cin, trx_no_s);
		if (TSEUtil::IsNumber(trx_no_s))
		{
			std::stringstream ss;
			ss << trx_no_s;
			ss >> trx_no;
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	std::cout << "Type: ";
	std::getline(std::cin, process_type);

	while (true)
	{
		std::cout << "Process Data: ";
		std::getline(std::cin, process_data);

		if ((process_data.length() >= 3) && (process_data.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(process_data.substr(3), process_data_bin);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				process_data_ptr = process_data_bin.data();
				process_data_len = (unsigned long) process_data_bin.size();
				break;
			}
		}//if
		else
		{
			process_data_ptr = (unsigned char *)process_data.c_str();
			process_data_len = (unsigned long) process_data.length();
			break;
		}
	}//while

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_UpdateTransaction(
		(unsigned char *)mCurrentUserId.c_str(),
		(unsigned long) mCurrentUserId.length(),
		trx_no,
		process_data_ptr,
		process_data_len,
		(unsigned char *)process_type.c_str(),
		(unsigned long) process_type.length(),
		&log_time,
		&signature,
		&signature_length,
		&signature_cntr);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	if (result == EXECUTION_OK)
	{
		struct tm stm;
		char log_time_s[sizeof("2019-01-01T23:59:59Z")] = "";
		time_t timet = log_time;

		if ((gmtime_s(&stm, &timet)) == 0)
		{
			strftime(log_time_s, sizeof(log_time_s), "%FT%TZ", &stm);
		}

		std::cout << "Transaction No : " << trx_no << std::endl;
		std::cout << "Log Time : " << log_time << " (" << log_time_s << ")" << std::endl;
		std::cout << "Signature Counter : " << signature_cntr << std::endl;
		std::cout << "Signature Value : ";
		for (unsigned long i = 0; i < signature_length; i++)
		{
			std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)signature[i] << " ";
		}
		std::cout << std::dec << std::endl;

		E_TSEMemDealloc(signature);
	}
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER)
	{
		mClientIdLogInList.remove(mCurrentUserId);
	}

	return result;
}

short int CuiLibCTester::CuiTimeAdminFinishTransaction()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string process_type = "";
	std::string process_data = "";
	std::string addtnl_data = "";
	std::string trx_no_s = "";
	unsigned long trx_no = 0;

	unsigned long log_time = 0;
	unsigned char* signature = NULL;
	unsigned long signature_length = 0;
	unsigned long signature_cntr = 0;

	unsigned char* process_data_ptr = NULL;
	unsigned long process_data_len = 0;
	unsigned char* additnl_data_ptr = NULL;
	unsigned long additnl_data_len = 0;
	std::vector<unsigned char> process_data_bin;
	std::vector<unsigned char> additnl_data_bin;

	if (IsTimeAdminLogin() == false)
	{
		return OTHER_ERROR;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[FINISH TRANSACTION]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	while (true)
	{
		std::cout << "Transaction No: ";
		std::getline(std::cin, trx_no_s);
		if (TSEUtil::IsNumber(trx_no_s))
		{
			std::stringstream ss;
			ss << trx_no_s;
			ss >> trx_no;
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	std::cout << "Type: ";
	std::getline(std::cin, process_type);
	while (true)
	{
		std::cout << "Process Data: ";
		std::getline(std::cin, process_data);
		if ((process_data.length() >= 3) && (process_data.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(process_data.substr(3), process_data_bin);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				process_data_ptr = process_data_bin.data();
				process_data_len = (unsigned long) process_data_bin.size();
				break;
			}
		}//if
		else
		{
			process_data_ptr = (unsigned char *)process_data.c_str();
			process_data_len = (unsigned long) process_data.length();
			break;
		}
	}//while

	while (true)
	{
		std::cout << "Additional Data: ";
		std::getline(std::cin, addtnl_data);
		if ((addtnl_data.length() >= 3) && (addtnl_data.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(addtnl_data.substr(3), additnl_data_bin);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				additnl_data_ptr = additnl_data_bin.data();
				additnl_data_len = (unsigned long) additnl_data_bin.size();
				break;
			}
		}//if
		else
		{
			additnl_data_ptr = (unsigned char *)addtnl_data.c_str();
			additnl_data_len = (unsigned long) addtnl_data.length();
			break;
		}
	}//while

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_FinishTransaction(
		(unsigned char *)mCurrentUserId.c_str(),
		(unsigned long) mCurrentUserId.length(),
		trx_no,
		process_data_ptr,
		process_data_len,
		(unsigned char *)process_type.c_str(),
		(unsigned long) process_type.length(),
		additnl_data_ptr,
		additnl_data_len,
		&log_time,
		&signature,
		&signature_length,
		&signature_cntr);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	if (result == EXECUTION_OK)
	{
		struct tm stm;
		char log_time_s[sizeof("2019-01-01T23:59:59Z")] = "";
		time_t timet = log_time;

		if ((gmtime_s(&stm, &timet)) == 0)
		{
			strftime(log_time_s, sizeof(log_time_s), "%FT%TZ", &stm);
		}

		std::cout << "Transaction No : " << trx_no << std::endl;
		std::cout << "Log Time : " << log_time << " (" << log_time_s << ")" << std::endl;
		std::cout << "Signature Counter : " << signature_cntr << std::endl;
		std::cout << "Signature Value : ";
		for (unsigned long i = 0; i < signature_length; i++)
		{
			std::cout << std::setfill('0') << std::setw(2) << std::hex << (int)signature[i] << " ";
		}
		std::cout << std::dec << std::endl;

		E_TSEMemDealloc(signature);
	}
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER)
	{
		mClientIdLogInList.remove(mCurrentUserId);
	}

	return result;
}

short int CuiLibCTester::CuiHost()
{
	short int result = 0;
	std::string sel_opt = "";
	int n_opt = -1;

	while (true)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[OPERATE AS HOST]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		std::cout << "Please select the operation you want to perform" << std::endl;

		std::cout << "[1] Unblock User for Admin" << std::endl;
		std::cout << "[2] Unblock User for TimeAdmin" << std::endl;
		std::cout << "[3] Change PUK" << std::endl;

		std::cout << "[99] Back" << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> sel_opt;

		if (TSEUtil::IsNumber(sel_opt))
		{
			std::stringstream ssTmp(sel_opt);
			ssTmp >> n_opt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}

		if (n_opt == 99)
		{
			break;
		}
	
		switch (n_opt)
		{
			case 1: result = CuiHostUnblockAdmin();  break;
			case 2: result = CuiHostUnblockTimeAdmin();  break;
			case 3: result = CuiHostChangePuk();  break;
			case 99: break;
			default:
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
		}
	}

	return EXECUTION_OK;
}

short int CuiLibCTester::CuiHostUnblockAdmin()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string puk = "";
	std::string new_pin = "";
	std::string secret_key = "";
	std::string user_id = EPSON_TSE_ADMIN;
	short int remaining_retries = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UNBLOCK ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "PUK: ";
	std::getline(std::cin, puk);
	std::cout << "New PIN: ";
	std::getline(std::cin, new_pin);
	std::cout << "Secret Key: ";
	std::getline(std::cin, secret_key);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_UnblockUser(
		(unsigned char *)user_id.c_str(),
		(unsigned long) user_id.length(),
		(unsigned char *)puk.c_str(),
		(unsigned long) puk.length(),
		(unsigned char *)new_pin.c_str(),
		(unsigned long) new_pin.length(),
		(unsigned char *)secret_key.c_str(),
		(unsigned long) secret_key.length(),
		&remaining_retries
		);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Remaining retries : " << remaining_retries << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return result;
}


short int CuiLibCTester::CuiHostUnblockTimeAdmin()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string puk = "";
	std::string new_pin = "";
	std::string secret_key = "";
	std::string user_id = "";
	short int remaining_retries = 0;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UNBLOCK TIME ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "Client Id: ";
	std::getline(std::cin, user_id);
	std::cout << "PUK: ";
	std::getline(std::cin, puk);
	std::cout << "New PIN: ";
	std::getline(std::cin, new_pin);
	std::cout << "Secret Key: ";
	std::getline(std::cin, secret_key);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_UnblockUser(
		(unsigned char *)user_id.c_str(),
		(unsigned long) user_id.length(),
		(unsigned char *)puk.c_str(),
		(unsigned long) puk.length(),
		(unsigned char *)new_pin.c_str(),
		(unsigned long) new_pin.length(),
		(unsigned char *)secret_key.c_str(),
		(unsigned long) secret_key.length(),
		&remaining_retries
		);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Remaining retries : " << remaining_retries << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return result;
}

short int CuiLibCTester::CuiHostChangePuk()
{
	short int status_code = EXECUTION_OK;
	short int result = EXECUTION_OK;
	unsigned char result_desc[100];

	std::string old_puk = "";
	std::string new_puk = "";
	std::string secret_key = "";
	short int remaining_retries;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[CHANGE PUK]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "Old PUK: ";
	std::getline(std::cin, old_puk);
	std::cout << "New PUK: ";
	std::getline(std::cin, new_puk);
	std::cout << "Secret Key: ";
	std::getline(std::cin, secret_key);

	if ((status_code = E_TSEOpenDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	result = E_ChangePuk(
		(unsigned char *)old_puk.c_str(),
		(unsigned long) old_puk.length(),
		(unsigned char *)new_puk.c_str(),
		(unsigned long) new_puk.length(),
		(unsigned char *)secret_key.c_str(),
		(unsigned long) secret_key.length(),
		&remaining_retries);

	if ((status_code = E_TSECloseDevice()) != EXECUTION_OK)
	{
		return status_code;
	}

	E_TSEGetErrorDescription(result, mErrorDescLang, result_desc);
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl;
	std::cout << "RESULT : " << result << std::endl;
	std::cout << "RESULT Description : " << result_desc << std::endl;
	std::cout << "Remaining retries : " << remaining_retries << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return result;
}

bool CuiLibCTester::IsAdminLogin()
{
	return mIsAdminLogin;
}

bool CuiLibCTester::IsTimeAdminLogin()
{
	auto it = std::find(mClientIdLogInList.begin(), mClientIdLogInList.end(), mCurrentUserId);
	return (it != mClientIdLogInList.end());
}

bool CuiLibCTester::GetHexString(std::string hexString, std::vector<unsigned char> &binData)
{
	binData.clear();

	if ((hexString.length() == 0) || ((hexString.length() % 2) != 0))
	{
		return false;
	}
	for (unsigned long i = 0; (i+1) < hexString.length(); i += 2)
	{
		const char chr1 = hexString.at(i);
		const char chr2 = hexString.at(i+1);

		unsigned char b1;
		unsigned char b2;

		unsigned char byteS = 0;

		if (chr1 >= '0' && chr1 <= '9')
		{
			b1 = chr1 - '0';
		}
		else if (chr1 >= 'a' && chr1 <= 'f')
		{
			b1 = chr1 - 'a' + 10;
		}
		else if (chr1 >= 'A' && chr1 <= 'F')
		{
			b1 = chr1 - 'A' + 10;
		}
		else
		{
			return false;
		}

		if (chr2 >= '0' && chr2 <= '9')
		{
			b2 = chr2 - '0';
		}
		else if (chr2 >= 'a' && chr2 <= 'f')
		{
			b2 = chr2 - 'a' + 10;
		}
		else if (chr2 >= 'A' && chr2 <= 'F')
		{
			b2 = chr2 - 'A' + 10;
		}
		else
		{
			return false;
		}

		byteS = (b1 * 16) + b2;
		binData.push_back(byteS);
	}
	return true;
}


//EOF