package example.printerscanner;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import android.app.Activity;
import android.os.Bundle;
import android.os.Handler;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

public class MainActivity extends Activity {
	private static final int PORT = 8009;
	private static final boolean BUFFER_ENABLED = true;

	private Socket connection = null;
	private BufferedReader reader = null;
	private BufferedWriter writer = null;
	private Document doc = null;

	private String ipAddress;
	private String deviceIDprinter;
	private String deviceIDscanner;
	private boolean printerOpened = false;

	private TextView console;
	private EditText ipAddressText;
	private EditText printerIDText;
	private EditText scannerIDText;
	private EditText onDataText;
	private Button connectButton;
	private Button printButton;
	private Handler mHandler;

	private boolean connecting = false;
	private boolean disconnecting = false;
	private boolean reconnecting = false;
	private boolean canceledReconnection = false;
	private String clientID = null;
	private String dataID = null;

	/** Called when the activity is first created. */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);
		getWindow().setSoftInputMode(
				WindowManager.LayoutParams.SOFT_INPUT_STATE_ALWAYS_HIDDEN);

		console = (TextView) findViewById(R.id.textArea);
		ipAddressText = (EditText) findViewById(R.id.editText1);
		printerIDText = (EditText) findViewById(R.id.editText2);
		scannerIDText = (EditText) findViewById(R.id.editText3);
		onDataText = (EditText) findViewById(R.id.editText4);
		mHandler = new Handler();

		connectButton = (Button) findViewById(R.id.Button01);
		connectButton.setOnClickListener(new View.OnClickListener() {

			@Override
			public void onClick(View v) {
				if (connecting == false) {
					connecting = true;
					connectButton.setEnabled(false);
					new Thread(new Runnable() {

						@Override
						public void run() {
							connect();
						}
					}).start();
				}
			}
		});

		printButton = (Button) findViewById(R.id.Button02);
		printButton.setOnClickListener(new View.OnClickListener() {

			@Override
			public void onClick(View v) {
				new Thread(new Runnable() {

					@Override
					public void run() {
						print();
					}
				}).start();
			}
		});
	}

	/**
	 * Connect to ePOS-Device XML Server using Socket.
	 */
	public void connect() {
		ipAddress = ipAddressText.getText().toString();
		InetSocketAddress serverAddress = new InetSocketAddress(ipAddress, PORT);
		try {
			connection = new Socket();
			connection.connect(serverAddress, 5000);
		} catch (UnknownHostException e) {
			appendConsole("Connecting to server failed.");
			e.printStackTrace();
			connecting = false;
			connection = null;
			enableButton(true);
			return;
		} catch (IOException e) {
			appendConsole("Connecting to server failed.");
			e.printStackTrace();
			connecting = false;
			connection = null;
			enableButton(true);
			return;
		}

		try {
			reader = new BufferedReader(new InputStreamReader(
					connection.getInputStream()));
			writer = new BufferedWriter(new OutputStreamWriter(
					connection.getOutputStream()));

			connection.setSoTimeout(5000);

			// Recieve reply message from server
			int chr;
			StringBuffer buffer = new StringBuffer();
			while ((chr = reader.read()) != 0) {
				buffer.append((char) chr);
			}

			// Parse recieved xml document(DOM)
			DocumentBuilder builder = DocumentBuilderFactory.newInstance()
					.newDocumentBuilder();
			doc = builder.parse(new ByteArrayInputStream(buffer.toString()
					.getBytes("UTF-8")));
			String firstNode = doc.getFirstChild().getNodeName();

			// Response of connect request
			if (firstNode.equals("connect")) {
				appendConsole("Connect to server success.");

				if (reconnecting == false) {
					// Disconnect old connection
					if (clientID != null) {
						disconnect();
					}

					new Thread(new Runnable() {
						@Override
						public void run() {
							onReceive();
						}
					}).start();

					openPrinter();
					openScanner();
				}

				clientID = getChildren(doc, "client_id");

				// Change connect button to disconnect button
				connectButton.setOnClickListener(new View.OnClickListener() {
					@Override
					public void onClick(View v) {
						if (disconnecting == false) {
							disconnecting = true;
							enableButton(false);
							new Thread(new Runnable() {
								@Override
								public void run() {
									disconnect();
									closeSocket();
									appendConsole("Disconnected");
								}
							}).start();
						}
					}
				});

				setTexttoButton("disconnect");
				enableButton(true);
				connecting = false;
			} else {
				appendConsole("Connect to server failed.");
				closeSocket();
			}
		} catch (SocketTimeoutException e) {
			appendConsole("Read message timed out.");
			closeSocket();
			e.printStackTrace();
		} catch (IOException e) {
			appendConsole("Disconnected.");
			closeSocket();
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			appendConsole("XML parse error.");
			e.printStackTrace();
		} catch (SAXException e) {
			appendConsole("XML parse error.");
			e.printStackTrace();
		}
		connecting = false;
	}

	/**
	 * Send disconnect message to server.
	 */
	public void disconnect() {
		String req = "<disconnect>" + "<data>"
				+ "<client_id>" + clientID + "</client_id>"
				+ "</data>" + "</disconnect>" + "\0";

		try {
			writer.write(req);
			writer.flush();

		} catch (IOException e) {
			appendConsole("Disconnected");
			closeSocket();
			e.printStackTrace();
		}
	}

	/**
	 * Reconnect to server.
	 */
	public void reconnect() {
		reconnecting = true;
		String oldClinetID = clientID;
		String receivedDataID = dataID;

		appendConsole("Reconnecting.");

		// Cancel reconnection if connect button pushed
		connectButton.setOnClickListener(new View.OnClickListener() {
			@Override
			public void onClick(View v) {
				canceledReconnection = true;
				enableButton(false);
			}
		});

		closeSocket();

		do {
			try {
				Thread.sleep(5000);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
			connect();
		}
		while (connection == null && canceledReconnection == false);

		canceledReconnection = false;
		reconnecting = false;

		if (connection == null) {
			connectButton.setOnClickListener(new View.OnClickListener() {
				@Override
				public void onClick(View v) {
					if (connecting == false) {
						connecting = true;
						connectButton.setEnabled(false);
						new Thread(new Runnable() {
							@Override
							public void run() {
								connect();
							}
						}).start();
					}
				}
			});

			setTexttoButton("connect");
			enableButton(true);
			appendConsole("Canceled reconnection.");
		}
		else {
			String req = "<reconnect>" + "<data>"
					+ "<new_client_id>" + clientID + "</new_client_id>"
					+ "<old_client_id>" + oldClinetID + "</old_client_id>"
					+ "<received_id>" + receivedDataID + "</received_id>"
					+ "</data>" + "</reconnect>" + "\0";

			try {
				writer.write(req);
				writer.flush();

			} catch (IOException e) {
				appendConsole("Disconnected");
				closeSocket();
				e.printStackTrace();
			}
		}
	}

	/**
	 * Close stream reader, writer and socket connection.
	 */
	public void closeSocket() {

		disconnecting = true;

		// Wait for onRecieve thread exits
		try {
			Thread.sleep(1000);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}

		try {
			if (reader != null) {
				reader.close();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		reader = null;

		try {
			if (writer != null) {
				writer.close();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		writer = null;

		try {
			if (connection != null) {
				connection.close();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		connection = null;

		printerOpened = false;

		if (reconnecting == false) {
			// Change disconnect button to connect button
			connectButton.setOnClickListener(new View.OnClickListener() {
				@Override
				public void onClick(View v) {
					if (connecting == false) {
						connecting = true;
						connectButton.setEnabled(false);
						new Thread(new Runnable() {
							@Override
							public void run() {
								connect();
							}
						}).start();
					}
				}
			});

			setTexttoButton("connect");
			enableButton(true);
		}

		disconnecting = false;
	}

	/**
	 * Receive XML message from the server and parse XML.
	 */
	public void onReceive() {
		int chr;
		StringBuffer buffer;

		try {
			connection.setSoTimeout(500);
		} catch (SocketException e) {
			e.printStackTrace();
		}

		// Read input stream until disconnected
		while ((disconnecting == false) && (connection != null)) {
			try {
				buffer = new StringBuffer();
				while ((chr = reader.read()) > 0) {
					buffer.append((char) chr);
				}

				// Parse recieved xml document(DOM)
				DocumentBuilder builder = DocumentBuilderFactory.newInstance()
						.newDocumentBuilder();
				doc = builder.parse(new ByteArrayInputStream(buffer.toString()
						.getBytes("UTF-8")));
				String firstNode = doc.getFirstChild().getNodeName();

				// Response of open_device request
				if (firstNode.equals("open_device")) {
					String id = getChildren(doc, "device_id");
					String code = getChildren(doc, "code");
					appendConsole(id + " : " + code);

					if (id.equals(deviceIDprinter) && code.equals("OK")) {
						printerOpened = true;
					}
				}

				// Server and client exchange the data using device_data message.
				else if (firstNode.equals("device_data")) {

					// Input data from scanner
					if (getChildren(doc, "type").equals("ondata")) {
						appendOnData(getChildren(doc, "input"));
					}

					// Response of print request from printer
					else if (getChildren(doc, "type").equals("onxmlresult")) {
						Element el = (Element) doc.getElementsByTagName(
								"response").item(0);
						if (el.getAttribute("success").equals("true")) {
							appendConsole("Print success.");
						} else {
							appendConsole("Print failed.");
						}
					}
				}

				// Response of reconnect request
				else if (firstNode.equals("reconnect")) {
					if (getChildren(doc, "code").equals("OK")) {
						printerOpened = true;
						appendConsole("Reconnected.");
					}
				}

				// Save latest number of data_id
				String latestDataID = getChildren(doc, "data_id");
				if (latestDataID != null) {
					dataID = latestDataID;
				}

			} catch (SocketTimeoutException e) {
			} catch (IOException e) {
				e.printStackTrace();
				appendConsole("Disconnected.");
				reconnect();
			} catch (SAXException e) {
				appendConsole("XML parse error.");
				e.printStackTrace();
			} catch (ParserConfigurationException e) {
				appendConsole("XML parse error.");
				e.printStackTrace();
			}
		}
	}

	/**
	 * Send open_device message to use printer.
	 */
	public void openPrinter() {
		deviceIDprinter = printerIDText.getText().toString();

		String req = "<open_device>" + "<device_id>" + deviceIDprinter
				+ "</device_id>" + "<data>" + "<type>type_printer</type>"
				+ "</data>" + "</open_device>" + "\0";

		try {
			writer.write(req);
			writer.flush();

		} catch (IOException e) {
			appendConsole("Disconnected");
			closeSocket();
			e.printStackTrace();
		}
	}

	/**
	 * Send open_device message to use scanner.
	 */
	public void openScanner() {
		deviceIDscanner = scannerIDText.getText().toString();

		String req = "<open_device>"
				+ "<device_id>" + deviceIDscanner + "</device_id>"
				+ "<data>" + "<type>type_scanner</type>"
				+ "<buffer>" + BUFFER_ENABLED + "</buffer>"
				+ "</data>" + "</open_device>" + "\0";

		try {
			writer.write(req);
			writer.flush();

		} catch (IOException e) {
			appendConsole("Disconnected.");
			closeSocket();
			e.printStackTrace();
		}
	}

	//
	/**
	 * Send print request.
	 */
	public void print() {
		if (printerOpened == false) {
			appendConsole("Printer is not opened.");
			return;
		}

		String text = onDataText.getText().toString();
		String req = "<device_data>"
				+ "<sequence>100</sequence>"
				+ "<device_id>"
				+ deviceIDprinter
				+ "</device_id>"
				+ "<data>"
				+ "<type>print</type>"
				+ "<timeout>10000</timeout>"
				+ "<printdata>"
				+ "<epos-print xmlns=\"http://www.epson-pos.com/schemas/2011/03/epos-print\">"
				+ "<text lang='ja' smooth='true'>Sample Print&#10;" + text
				+ "</text>"
				+ "<cut type=\"feed\" />"
				+ "</epos-print>" + "</printdata>" + "</data>"
				+ "</device_data>" + "\0";

		try {
			writer.write(req);
			writer.flush();
		} catch (IOException e) {
			appendConsole("Disconnected");
			closeSocket();
			e.printStackTrace();
		}
	}

	/**
	 * Get value of child node from specified Document object and TagName.
	 *
	 * @param doc specified Document object
	 * @param tagName specified xml tagname
	 * @return String of node value. If there is no such tagname, return null.
	 */
	public String getChildren(Document doc, String tagName) {
		NodeList list = doc.getElementsByTagName(tagName);
		if (list.getLength() == 0) {
			return null;
		} else {
			try {
				Node node = list.item(0);
				return node.getFirstChild().getNodeValue();
			}
			catch (NullPointerException e){
				return null;
			}
		}
	}


	/**
	 * Append text to textview for console.
	 *
	 * @param text String to add
	 */
	public void appendConsole(final String text) {
		mHandler.post(new Runnable() {
			@Override
			public void run() {
				String temp = console.getText().toString();
				console.setText(text + "\n" + temp);
			}
		});
	}

	//
	/**
	 * Append text to textview for ondata of scanner.
	 *
	 * @param text String to add
	 */
	public void appendOnData(final String text) {
		mHandler.post(new Runnable() {
			@Override
			public void run() {
				onDataText.append(text + "\n");
			}
		});
	}

	/**
	 * Enable or disable connectButton.
	 *
	 * @param flag True if this view is enabled, false otherwise
	 */
	public void enableButton(final boolean flag) {
		mHandler.post(new Runnable() {
			@Override
			public void run() {
				connectButton.setEnabled(flag);
			}
		});
	}

	/**
	 * Set string value of connectButton.
	 *
	 * @param text String to set
	 */
	public void setTexttoButton(final String text) {
		mHandler.post(new Runnable() {
			@Override
			public void run() {
				connectButton.setText(text);
			}
		});
	}

	@Override
	public void onPause() {
		super.onPause();
		if (connection != null) {
			new Thread(new Runnable() {
				@Override
				public void run() {
					disconnect();
					closeSocket();
					appendConsole("Disconnected");
				}
			}).start();
		}
	}
}
