// Copyright 2019-2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/11  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include "TSEAccessByEscPosSerial.h"
#include "AccessBySerial.h"
#include "TSEPrintLog.h"
#include <sstream>

#define DEFAULT_RCV_TIMEOUT		20000
#define LONGER_RCV_TIMEOUT		60000  //Set-Up tse and RunSelfTest may take longer

TSEAccessByEscPosSerial::TSEAccessByEscPosSerial(const std::string& tIdentifier)
{
	mSerial = NULL;
	try
	{
		mSerial = new AccessSerial(tIdentifier.c_str());
	}
	catch (...)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Creating Serial Port Failed."));
		throw std::runtime_error("");
	}
}

TSEAccessByEscPosSerial::~TSEAccessByEscPosSerial()
{
	delete mSerial;
	mSerial = NULL;
}

bool TSEAccessByEscPosSerial::EnableTseAccess()
{
	unsigned long rcvTime = 0;

	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd{0x10, 0x14, 0x06, 0x01, 0x03, 0x01, 0x03, 0x14, 0x01, 0x06, 0x02, 0x08};
	
	mTotalRcvTime = 0;
	mTotalTime = 0;
	SYSTEMTIME st, et;
	union timeunion {
		FILETIME fT;
		ULARGE_INTEGER uL;
	} fst, fet;
	std::stringstream sLogAll;
	GetSystemTime(&st);
	SystemTimeToFileTime(&st, &fst.fT);
	try
	{
		SendMsg(sendCmd);
		while (true)
		{
			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Enabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x37) continue;
			mTotalRcvTime += rcvTime;

			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Enabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x5C) continue;
			mTotalRcvTime += rcvTime;
			break;
		}

		RecvMsg(recvBuffer, 2, DEFAULT_RCV_TIMEOUT, rcvTime);
		mTotalRcvTime += rcvTime;
	}
	catch (int&)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Enabling TSE Access Failed."));
		throw std::runtime_error("");
	}
	catch (...)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Enabling TSE Access Failed. Other error."));
		throw std::runtime_error("");
	}

	if (recvBuffer[0] != 0x30)
	{
		return false;
	}

	GetSystemTime(&et);
	SystemTimeToFileTime(&et, &fet.fT);
	mTotalTime = (unsigned long)((fet.uL.QuadPart - fst.uL.QuadPart) / 10000);

	return true;
}

void TSEAccessByEscPosSerial::SendJsonStringToTse(const std::string& tJsonStr, std::string& tResponse)
{
	unsigned long rcvTime = 0;

	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd{ 0x1C, 0x28, 0x50, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00 };
	std::vector<BYTE> sendAck{ 0x06 };
	std::vector<BYTE> sendCmdJson;

	unsigned long jsonSize = (unsigned long) tJsonStr.length();

	sendCmd[3] = (jsonSize + 4) & 0xFF;				//set pL
	sendCmd[4] = ((jsonSize + 4) >> 8) & 0xFF;		//set pH
	int totalsize = sizeof(sendCmd) + jsonSize;
	sendCmdJson.insert(sendCmdJson.end(), sendCmd.data(), sendCmd.data() + sendCmd.size());
	sendCmdJson.insert(sendCmdJson.end(), tJsonStr.c_str(), tJsonStr.c_str() + jsonSize);

	tResponse = "";
	mTotalRcvTime = 0;	
	mTotalTime = 0;
	SYSTEMTIME st, et;
	union timeunion {
		FILETIME fT;
		ULARGE_INTEGER uL;
	} fst, fet;
	std::stringstream sLogAll;
	GetSystemTime(&st);
	SystemTimeToFileTime(&st, &fst.fT);
	try
	{
		SendMsg(sendCmdJson);
		BYTE status = 0;
		do
		{
			while (true)
			{
				RecvMsg(recvBuffer, 1, LONGER_RCV_TIMEOUT, rcvTime);
				if (recvBuffer.size() != 1)
				{
					TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
					throw std::runtime_error("");
				}
				if (recvBuffer[0] != 0x53) continue;
				mTotalRcvTime += rcvTime;
				
				RecvMsg(recvBuffer, 1, LONGER_RCV_TIMEOUT, rcvTime);
				if (recvBuffer.size() != 1)
				{
					TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
					throw std::runtime_error("");
				}
				if (recvBuffer[0] != 0x27) continue;
				mTotalRcvTime += rcvTime;
				break;
			}

			RecvMsg(recvBuffer, 4, LONGER_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 4)
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
				throw std::runtime_error("");
			}

			mTotalRcvTime += rcvTime;
			status = recvBuffer[0];
			unsigned short bsize = (((unsigned short)recvBuffer[3]) << 8) | recvBuffer[2];

			RecvMsg(recvBuffer, bsize, LONGER_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != bsize)
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
				throw std::runtime_error("");
			}

			mTotalRcvTime += rcvTime;
			SendMsg(sendAck);
			tResponse.append((char *)recvBuffer.data(), bsize);
		} while (status != 0x40);
	}
	catch (int&)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed."));
		throw std::runtime_error("");
	}
	catch (...)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Sending JSON data Failed. Other error."));
		throw std::runtime_error("");
	}

	GetSystemTime(&et);
	SystemTimeToFileTime(&et, &fet.fT);
	mTotalTime = (unsigned long)((fet.uL.QuadPart - fst.uL.QuadPart) / 10000);

}

bool TSEAccessByEscPosSerial::DisableTseAccess()
{
	unsigned long rcvTime = 0;

	std::vector<BYTE> recvBuffer;
	std::vector<BYTE> sendCmd{ 0x10, 0x14, 0x06, 0x01, 0x01, 0x01, 0x03, 0x14, 0x01, 0x06, 0x02, 0x08 };

	mTotalRcvTime = 0;
	mTotalTime = 0;
	SYSTEMTIME st, et;
	union timeunion {
		FILETIME fT;
		ULARGE_INTEGER uL;
	} fst, fet;
	std::stringstream sLogAll;
	GetSystemTime(&st);
	SystemTimeToFileTime(&st, &fst.fT);
	try
	{
		SendMsg(sendCmd);

		while (true)
		{
			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x37) continue;
			mTotalRcvTime += rcvTime;

			RecvMsg(recvBuffer, 1, DEFAULT_RCV_TIMEOUT, rcvTime);
			if (recvBuffer.size() != 1)
			{
				TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed."));
				throw std::runtime_error("");
			}
			if (recvBuffer[0] != 0x5C) continue;
			mTotalRcvTime += rcvTime;
			break;
		}
		RecvMsg(recvBuffer, 2, DEFAULT_RCV_TIMEOUT, rcvTime);
		if (recvBuffer.size() != 2)
		{
			TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed."));
			throw std::runtime_error("");
		}
		mTotalRcvTime += rcvTime;
	}
	catch (int&)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed."));
		throw std::runtime_error("");
	}
	catch (...)
	{
		TSEPRINTLOG(LIBLOG_LEVEL_ERROR, ("Disabling TSE Access Failed. Other error."));
		throw std::runtime_error("");
	}

	if (recvBuffer[0] != 0x30)
	{
		return false;
	}

	GetSystemTime(&et);
	SystemTimeToFileTime(&et, &fet.fT);
	mTotalTime = (unsigned long)((fet.uL.QuadPart - fst.uL.QuadPart) / 10000);

	return true;
}

void TSEAccessByEscPosSerial::SendTo(const std::vector<unsigned char>& tBuffer)
{
	mSerial->SerialSend(tBuffer);
}

void TSEAccessByEscPosSerial::RecvFrom(std::vector<BYTE>& tBuffer, const unsigned long& tMaxSize, const unsigned long& tTimeout)
{
	mSerial->SerialRecv(tBuffer, tMaxSize, tTimeout);
}

//EOF