// Copyright 2019-2021 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2021/03  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include <windows.h>
#include <wincrypt.h>
#include "TSEUtil.h"
#include <ctime>
#include <fstream>
#include <regex>
#include <algorithm>
#include <cmath>
#include <cctype>
#undef USING_REGEX

std::string TSEUtil::GetCurrentUtcTime() {
	time_t rawtime = std::time(nullptr);
	tm timeinfo;
	char buffer[sizeof("2019-01-01T23:59:59Z")];
	gmtime_s(&timeinfo, &rawtime);
	strftime(buffer, sizeof(buffer), "%FT%TZ", &timeinfo);
	return buffer;
}

void TSEUtil::DecodeBase64(const std::string& tString, std::vector<unsigned char>& tDecoded)
{
	std::string tmpStr = tString;
	tmpStr.erase(std::remove(tmpStr.begin(), tmpStr.end(), '\\'), tmpStr.end());

	unsigned long decodeLength = (unsigned long) tmpStr.size();
	unsigned char *tmpDecoded = new unsigned char[tmpStr.size()];
	
	if ( CryptStringToBinary(
			tmpStr.c_str(),
			decodeLength,
			CRYPT_STRING_BASE64,
			tmpDecoded,
			&decodeLength,
			NULL,
			NULL) == 0)
	{
		delete tmpDecoded;
		throw std::runtime_error("Failed to decode the string [1].");
	}

	std::copy(tmpDecoded, (tmpDecoded + decodeLength), std::back_inserter(tDecoded));
	delete tmpDecoded;
}

std::string TSEUtil::EncodeBase64(const std::vector<unsigned char>& tData)
{
	char * Base64 = NULL;
	unsigned long Base64Size = (unsigned long)(std::ceil(tData.size() / 3)) * 4; //aprox size

	//Get the size needed for the the expected encoded stream
	if (!CryptBinaryToString(
		(const BYTE *)tData.data(),
		(DWORD) tData.size(),
		//CRYPT_STRING_BASE64,
		CRYPT_STRING_BASE64 | CRYPT_STRING_NOCRLF,
		NULL,
		&Base64Size))
	{
		throw std::runtime_error("Failed to encode the string [1].");
	}

	Base64 = new char[Base64Size];
	//Generates a Base64 stream
	if (!CryptBinaryToString(
		(const BYTE *)tData.data(),
		(DWORD) tData.size(),
		//CRYPT_STRING_BASE64,
		CRYPT_STRING_BASE64 | CRYPT_STRING_NOCRLF,
		(char *)Base64,
		&Base64Size))
	{
		delete Base64;
		throw std::runtime_error("Failed to encode the string [2].");
	}

	std::string encodedStr(Base64);
	encodedStr.erase(std::remove(encodedStr.begin(), encodedStr.end(), 0x0a), encodedStr.end());
	encodedStr.erase(std::remove(encodedStr.begin(), encodedStr.end(), 0x0d), encodedStr.end());

	std::vector<unsigned char> decode_check;
	DecodeBase64(encodedStr.c_str(), decode_check);

	if (decode_check.size() != tData.size())
	{
		throw std::runtime_error("Failed to encode the string [3].");
	}

	for (unsigned long i = 0; i < (unsigned long) tData.size(); i++)
	{
		if (tData[i] != decode_check[i])
		{
			throw std::runtime_error("Failed to encode the string [4].");
		}
	}

	delete Base64;
	return encodedStr;
}

std::string TSEUtil::EncodeBase64Sha256(const std::vector<unsigned char>& tData)
{
	HCRYPTPROV hCryptProv;
	HCRYPTHASH hHash;
	unsigned char HashBytes[32] = { 0 };
	char Base64[128] = { 0 };
	unsigned long HashSize = 32;
	unsigned long Base64Size = 128;

	// Handle to a cryptography provider context.
	if (! CryptAcquireContext(
			&hCryptProv,
			NULL,
			NULL,
			PROV_RSA_AES, 
			CRYPT_VERIFYCONTEXT))
	{
		throw std::runtime_error("Failed to encode the string [1].");
	}

	// Initiate hashing using SHA256
	if (! CryptCreateHash(
			hCryptProv,
			CALG_SHA_256,
			0,
			0,
			&hHash))
	{
		throw std::runtime_error("Failed to encode the string [2].");
	}

	//Adds data to the hash object.
	if (!CryptHashData(
			hHash, 
			tData.data(), 
			(DWORD) tData.size(), 
			0))
	{
		CryptDestroyHash(hHash);
		CryptReleaseContext(hCryptProv, 0);
		throw std::runtime_error("Failed to encode the string [3].");
	}

	//Retrieves the hashed stream
	if (!CryptGetHashParam(hHash, HP_HASHVAL, HashBytes, &HashSize, 0))
	{
		CryptDestroyHash(hHash);
		CryptReleaseContext(hCryptProv, 0);
		return "";
	}

	//Generates a Base64 stream
	if (! CryptBinaryToString(
			(const BYTE *)HashBytes,
			HashSize,
			//CRYPT_STRING_BASE64,
			CRYPT_STRING_BASE64 | CRYPT_STRING_NOCRLF,
			(char *)Base64,
			&Base64Size))
	{
		throw std::runtime_error("Failed to encode the string [4].");
	}
	
	std::vector<unsigned char> decode_check;
	DecodeBase64(Base64, decode_check);

	if (decode_check.size() != HashSize)
	{
		throw std::runtime_error("Failed to encode the string [5].");
	}

	for (unsigned long i = 0; i < HashSize; i++)
	{
		if (HashBytes[i] != decode_check[i])
		{
			throw std::runtime_error("Failed to encode the string [6].");
		}
	}

	// Cleanup
	if (hHash)
		CryptDestroyHash(hHash);
	if (hCryptProv)
		CryptReleaseContext(hCryptProv, 0);

	std::string finalStr(Base64);
	finalStr.erase(std::remove(finalStr.begin(), finalStr.end(), 0x0a), finalStr.end());
	finalStr.erase(std::remove(finalStr.begin(), finalStr.end(), 0x0d), finalStr.end());

	return finalStr;

}

bool TSEUtil::SaveToFile(const std::string& tFilename, const unsigned char* tMsg, const unsigned long& tSize)
{
	if (tFilename != "")
	{
		std::ofstream tsePrintLogFile;
		tsePrintLogFile.open(tFilename, std::ofstream::out | std::ofstream::binary);

		if (tsePrintLogFile.is_open())
		{
			tsePrintLogFile.write((char *) tMsg, tSize);
			tsePrintLogFile.close();
		}
		else 
		{
			return false;
		}
	}

	return true;
}

bool TSEUtil::IsNumber(const std::string& tTestNum)
{
	std::string::const_iterator it = tTestNum.begin();

	while (it != tTestNum.end() && std::isdigit(*it))
	{
		++it;
	}
	return (!tTestNum.empty() && it == tTestNum.end());
}

bool TSEUtil::IsDateTime(const std::string& tTestNum)
{
#ifndef USING_REGEX
	if (tTestNum.length() != 20)
	{
		return false;
	}

	for (int index = 0; index < 20; index++)
	{
		if ((index == 4) || (index == 7)) /*-*/
		{
			if (tTestNum.at(index) != '-')
			{
				return false;
			}
		}
		else if (index == 10) /*T*/
		{
			if (tTestNum.at(index) != 'T')
			{
				return false;
			}
		}
		else if ((index == 13) || (index == 16)) /*:*/
		{
			if (tTestNum.at(index) != ':')
			{
				return false;
			}
		}
		else if (index == 19) /*Z*/
		{
			if (tTestNum.at(index) != 'Z')
			{
				return false;
			}
		}
		else
		{
			if (((tTestNum.at(index) >= '0') && (tTestNum.at(index) <= '9')) == false)
			{
				return false;
			}
		}
	}

	return true;
#else /*USING_REGEX*/
	std::string regstr("[0-9][0-9][0-9][0-9]-[0-9][0-9]-[0-9][0-9]T[0-9][0-9]:[0-9][0-9]:[0-9][0-9]Z");
	std::regex re(regstr);

	return std::regex_match(tTestNum, re);
#endif /*USING_REGEX*/
}

//Creates directory structure. Not the file.
bool TSEUtil::CreateTseDirectory(const std::string& tFileName)
{
	if (tFileName == "")
	{
		return false;
	}

	size_t found;
	std::string filename = tFileName;

	found = filename.find_last_of("/\\");

	std::string dir = filename.substr(0, found);
	std::string file = filename.substr(found + 1);

	found = 0;
	do
	{
		found = dir.find_first_of("\\/", found + 1);
		
		std::string fldr = dir.substr(0, found);
		struct stat info;

		if ((found == 2) && (fldr.length() == 2) && (fldr.at(1) == ':')) /*Drive*/
		{
			int drvType = GetDriveType(fldr.c_str());
			if ((drvType == DRIVE_NO_ROOT_DIR) || (drvType == DRIVE_UNKNOWN))
			{
				return false;
			}
			continue;
		}
		
		if (stat(fldr.c_str(), &info) == 0)
		{
			if (info.st_mode & _S_IFDIR)
			{
				continue; /*Directory already exists*/
			}
		}
		
		if (!(CreateDirectory(dir.substr(0, found).c_str(), NULL) || ERROR_ALREADY_EXISTS == GetLastError()))
		{
			return false;
		}
	} while (found != std::string::npos);

	return true;
}

std::string TSEUtil::GetProgramDataPath()
{
	std::string path;

	char szPath[4096];
	memset(szPath, '\0', 4096);

	if (SUCCEEDED(GetEnvironmentVariable("ALLUSERSPROFILE", szPath, 4096)))
	{
		path = std::string(szPath);
	}
	return path;
}


//EOF