// Copyright 2019-2021 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2021/03  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include <windows.h>
#include <iostream>
#include <sstream>
#include <iomanip>
#include <regex>
#include "CuiOperateTSE.h"
#include "../../Core/TSEOperate.h"
#include "../../Core/TSESimpleJsonAndXmlParser.h"
#include "../../Core/TSEUtil.h"
#include "../../Core/TSEDeviceList.h"


#define MENU_COLOR_HEADER  (FOREGROUND_RED  | FOREGROUND_INTENSITY)
#define MENU_COLOR_RESULT  (FOREGROUND_BLUE | FOREGROUND_INTENSITY)
#define MENU_COLOR_DEFAULT (FOREGROUND_RED  | FOREGROUND_GREEN | FOREGROUND_BLUE)
#define MENU_COLOR_WARNING (FOREGROUND_RED  | FOREGROUND_GREEN)

#define EPSON_TSE_ADMIN		"Administrator"
#define RCVTIMELABEL		"Receive Time(ms)="
#define EXPORTFILEEXT		".tar"

CuiOperateTSE::CuiOperateTSE(const TSEOptions& tTseOpt):mLastExportFile("")
{
	mTseOpr = NULL;
	mIsAdminLogin = false;
	mIsHostAuthenticated = false;
	mIsTseInitialized = false;

	mConsole = GetStdHandle(STD_OUTPUT_HANDLE);
}

CuiOperateTSE::~CuiOperateTSE() 
{
}

bool CuiOperateTSE::ConnectToTse(const TSEOptions& tTseOpt)
{
	try 
	{
		mTseOpr = new TSEOperate(tTseOpt);
	}
	catch (std::runtime_error&)
	{
		mTseOpr = NULL;
		std::cout << "Connection error." << std::endl;
		return false;
	}
	return true;
}

bool CuiOperateTSE::DisconnectFromTse()
{
	delete mTseOpr;
	mTseOpr = NULL;

	return true;
}


bool CuiOperateTSE::CuiGetTseId(const std::string &tIpAddress, std::string& tTseId)
{
	std::string resultAll = "";
	std::string result = "";

	std::string tseList = "";
	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET TSE DEVICE LIST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	try
	{
		TSEGetDeviceList(tIpAddress, tseList);
	}
	catch (std::runtime_error&)
	{
		result = "OTHER_ERROR_CANNOT_GET_TSE_LIST";
	}

	if (result == "OTHER_ERROR_CANNOT_GET_TSE_LIST")
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "Cannot retrieve the TSE device list. Trying TSE=\"local_TSE\"..." << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		tTseId = "local_TSE";
	}
	else
	{
		std::vector<std::string> tokens;
		LStringToVector(tseList, tokens);

		unsigned short tse_cnt = 0;
		std::string tseId = "";

		resultAll += "\t tseList=";
		for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
		{
			resultAll += "\n\t\t" + *str_iter + ", ";
			tseId = *str_iter;
			tse_cnt++;
		}

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		if ((tse_cnt == 1) && (tseId == "local_TSE"))
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "Trying TSE=\"" << tseId << "\"..." << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

			tTseId = tseId;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			std::cout << "To exit, type bye. To connect to a TSE, enter the TSE ID: ";
			std::getline(std::cin, tTseId);
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	return true;
}

bool CuiOperateTSE::ShowMenu() 
{	
	std::string result = "";
	std::string tseInfo = "";
	std::string initializeStatus = "";
	std::string selOpt = "";
	int nOpt = -1;
	
	try
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->GetStorageInfo(tseInfo);
		mTseOpr->TSECloseDevice();
	}
	catch (std::runtime_error&)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
		std::cout << std::endl << "Runtime Error>> " << std::endl;
		return false;
	}

	initializeStatus = TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "tseInitializationState");

	if ((result == "EXECUTION_OK") || (result == "OTHER_ERROR_CURRENTLY_EXPORTING") || (result == "OTHER_ERROR_TSE_ALREADY_SET_UP"))
	{
		if (initializeStatus == "INITIALIZED") 
		{
			mIsTseInitialized = true;
		}

		if ((result == "OTHER_ERROR_CURRENTLY_EXPORTING") || (result == "OTHER_ERROR_TSE_ALREADY_SET_UP"))
		{
			mIsTseInitialized = true;
		}
	
		while (true)
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
			std::cout << std::endl << "[MAIN MENU]" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			std::cout << "TSE INITIALIZED? : " << (mIsTseInitialized ? "Yes" : "No") << std::endl;
			if (result != "EXECUTION_OK")
			{
				std::cout << "Other Status : " << result << std::endl;
			}
			std::cout << std::endl;

			std::cout << "Please select the operation you want to perform" << std::endl;
			std::cout << "[1] Get TSE Info" << std::endl;
			if (mIsTseInitialized)
			{
				std::cout << "[2] Operate as Admin" << std::endl;
				std::cout << "[3] Operate as TimeAdmin" << std::endl;
				std::cout << "[4] Operate as Host" << std::endl;
				std::cout << "[5] Run TSE Self Test" << std::endl;
				std::cout << "[6] Get Log Message Certificate" << std::endl;
				std::cout << "[7] Get Serial Number" << std::endl;
				std::cout << "[12] Get Last Transaction Response" << std::endl;
				std::cout << "[13] Get Started Transaction List" << std::endl;
				std::cout << "[14] Get Storage - Smart Information" << std::endl;
				std::cout << "[98] Factory Reset" << std::endl;
			}
			else
			{
				std::cout << "[2] Set-up TSE" << std::endl;
				std::cout << "[5] Run TSE Self Test" << std::endl;
			}
		
			std::cout << "[99] Exit Menu" << std::endl;
			std::cout << std::endl << "Enter only the number: ";
			std::cin >> selOpt;

			if (TSEUtil::IsNumber(selOpt))
			{
				std::stringstream ssTmp(selOpt);
				ssTmp >> nOpt;
			}
			else
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}

			if (nOpt == 99)
			{
				return false;
			}

			try 
			{
				if (mIsTseInitialized)
				{
					switch (nOpt)
					{
						case 1: CuiGetStorageInfo(); break;
						case 2: CuiAdmin(); break;
						case 3: CuiTimeAdmin();  break;
						case 4: CuiHost(); break;
						case 5: CuiRunTseSelfTest(); break;
						case 6: CuiGetLogMsgCertificate(); break;
						case 7: CuiExportSerialNumber(); break;
						case 12: CuiGetLastTrxResponse();  break;
						case 13: CuiGetStartedTrxList(); break;
						case 14: CuiGetStorageSmartInfo(); break;
						case 98: return CuiFactoryReset();
						default:
						{
							SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
							std::cout << "INVALID INPUT!" << std::endl;
							SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
							continue;
						}
					}
				}
				else
				{
					switch (nOpt)
					{
						case 1: CuiGetStorageInfo(); break;
						case 2: CuiSetup(); break;
						case 5: CuiRunTseSelfTest(); break;
						default:
						{
							SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
							std::cout << "INVALID INPUT!" << std::endl;
							SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
							continue;
						}
					}
				}
			}
			catch (std::runtime_error&) 
			{
				std::cout << "Error while executing option [" << nOpt << "]:" << std::endl;
				return false;
			}
		}//while
	}//if (result == "EXECUTION_OK") 
	std::cout << result << std::endl;
	return false;
}

void CuiOperateTSE::CuiAdmin()
{
	std::string result = "";
	std::string selOpt = "";
	int nOpt = -1;

	mCurrentUserId = EPSON_TSE_ADMIN;

	while (true)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[OPERATE AS ADMIN]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		std::cout << "LOGGED-IN? : " << (mIsAdminLogin ? "Yes" : "No") << std::endl << std::endl;

		std::cout << "Please select the operation you want to perform" << std::endl;
		if (mIsAdminLogin)
		{
			std::cout << "[1] Log-out" << std::endl;
			std::cout << "[2] Update Time For First" << std::endl;
			std::cout << "[3] Update Time" << std::endl;
			std::cout << "[4] Set Timeout Interval" << std::endl;
			std::cout << "[5] Get Timeout Interval" << std::endl;
			std::cout << "[6] Enable Export If Csp Test Fails" << std::endl;
			std::cout << "[7] Disable Export If Csp Test Fails" << std::endl;
			std::cout << "[8] Set Up For Printer" << std::endl;

			std::cout << "[10] Get Export Data (Archive Export)" << std::endl;
			std::cout << "[11] Get Export Data (Export Filtered By Transaction Number)" << std::endl;
			std::cout << "[12] Get Export Data (Export Filtered By Transaction Number Interval)" << std::endl;
			std::cout << "[13] Get Export Data (Export Filtered By Period Of Time)" << std::endl;
			std::cout << "[14] Finalize Export" << std::endl;
			std::cout << "[15] Cancel Export" << std::endl;

			std::cout << "[20] Register Shared Secret Key" << std::endl;
			std::cout << "[21] Register Client Id" << std::endl;
			std::cout << "[22] Deregister Client Id" << std::endl;
			std::cout << "[23] Unlock TSE" << std::endl;
			std::cout << "[24] Lock TSE" << std::endl;
			std::cout << "[25] Disable Secure Element" << std::endl;
			std::cout << "[26] Change Admin PIN" << std::endl;

			std::cout << "[30] Get Authenticated User List" << std::endl;
			std::cout << "[31] Get Registered Client Id List" << std::endl;

			std::cout << "[98] Update Time (Manual)" << std::endl;
		}
		else
		{
			std::cout << "[1] Log-in" << std::endl;
		}
		std::cout << "[99] Exit Menu" << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> selOpt;
		
		if (TSEUtil::IsNumber(selOpt))
		{
			std::stringstream ssTmp(selOpt);
			ssTmp >> nOpt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}
		
		if (nOpt == 99)
		{
			break;
		}

		try
		{
			switch (nOpt)
			{
				case 1: 
				{
					if (mIsAdminLogin) 
						CuiAdminLogout();
					else 
						CuiAdminLogin();
					break;
				}
				case 2: CuiUpdateTimeFirstTime();  break;
				case 3: CuiUpdateTime();  break;
				case 4: CuiAdminSetTimeout(); break;
				case 5: CuiAdminGetTimeout(); break;
				case 6: CuiAdminEnableExportIfCspTestFails(); break;
				case 7: CuiAdminDisableExportIfCspTestFails(); break;
				case 8: CuiAdminSetUpForPrinter(); break;

				case 10: CuiAdminGetExportDataArchive(); break;
				case 11: CuiAdminGetExportDataFilterByTrx(); break;
				case 12: CuiAdminGetExportDataFilterByTrxNoInterval(); break;
				case 13: CuiAdminGetExportDataFilterByPeriod(); break;
				case 14: CuiAdminFinalizeExport(); break;
				case 15: CuiAdminCancelExportData(); break;

				case 20: CuiAdminRegisterSecretKey();  break;
				case 21: CuiAdminRegisterClient();  break;
				case 22: CuiAdminDeregisterClient();  break;
				case 23: CuiAdminUnlockTse();  break;
				case 24: CuiAdminLockTse();  break;
				case 25: CuiAdminDisableSecureElement();  break;
				case 26: CuiChangeUserPin();  break;

				case 30: CuiAdminGetAuthenticatedUserList();  break;
				case 31: CuiAdminGetRegisteredClientList();  break;

				case 98: CuiUpdateTimeManually(); break;
				case 99: break;
				default: 
				{
					SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
					std::cout << "INVALID INPUT!" << std::endl;
					SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
					continue;
				}
			}
		}
		catch (std::runtime_error&)
		{
			std::cout << "Error while executing option [" << nOpt << "]:" << std::endl;
			mTseOpr->TSECloseDevice();
		}//try catch
	}//while (true)
}

void CuiOperateTSE::CuiGetStorageInfo()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string tseInfo = "";
	std::string yORn = "n";
	
	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET STORAGE INFO]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << std::endl << "Get TSE Info in raw JSON string format (y/n): ";
	std::cin >> yORn;

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetStorageInfo(tseInfo);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetStorageInfo() : " + result + "\n";

	if (result == "EXECUTION_OK")
	{
		if (yORn == "y")
		{
			resultAll += "\t Raw JSON string=\n";
			resultAll += "\t " + tseInfo;
		}
		else
		{
			resultAll += "\t [TSE INFO]\n";
			resultAll += "\t cdcId=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "cdcId") + "\"\n";
			resultAll += "\t cdcHash=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "cdcHash") + "\"\n";
			resultAll += "\t certificateExpirationDate=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "certificateExpirationDate") + "\"\n";
			resultAll += "\t createdSignatures=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "createdSignatures") + "\"\n";
			resultAll += "\t hardwareVersion=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "hardwareVersion") + "\"\n";
			resultAll += "\t hasPassedSelfTest=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "hasPassedSelfTest") + "\"\n";
			resultAll += "\t hasValidTime=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "hasValidTime") + "\"\n";
			resultAll += "\t isTransactionInProgress=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "isTransactionInProgress") + "\"\n";
			resultAll += "\t isExportEnabledIfCspTestFails=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "isExportEnabledIfCspTestFails") + "\"\n";
			resultAll += "\t isTSEUnlocked=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "isTSEUnlocked") + "\"\n";
			resultAll += "\t lastExportExecutedDate=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "lastExportExecutedDate") + "\"\n";
			resultAll += "\t maxRegisteredClients=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "maxRegisteredClients") + "\"\n";
			resultAll += "\t maxSignatures=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "maxSignatures") + "\"\n";
			resultAll += "\t maxStartedTransactions=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "maxStartedTransactions") + "\"\n";
			resultAll += "\t maxUpdateDelay=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "maxUpdateDelay") + "\"\n";
			resultAll += "\t registeredClients=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "registeredClients") + "\"\n";
			resultAll += "\t remainingSignatures=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "remainingSignatures") + "\"\n";
			resultAll += "\t signatureAlgorithm=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "signatureAlgorithm") + "\"\n";
			resultAll += "\t softwareVersion=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "softwareVersion") + "\"\n";
			resultAll += "\t startedTransactions=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "startedTransactions") + "\"\n";
			resultAll += "\t tarExportSize=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "tarExportSize") + "\"\n";
			resultAll += "\t timeUntilNextSelfTest=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "timeUntilNextSelfTest") + "\"\n";
			resultAll += "\t tseCapacity=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "tseCapacity") + "\"\n";
			resultAll += "\t tseCurrentSize=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(tseInfo, "tseCurrentSize") + "\"\n";
			resultAll += "\t tseDescription=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "tseDescription") + "\"\n";
			resultAll += "\t tseInitializationState=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "tseInitializationState") + "\"\n";
			resultAll += "\t vendorType=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "vendorType") + "\"\n";

			{
				std::vector<unsigned char> decoded_stream_ser;
				std::stringstream ss;
				try
				{
					TSEUtil::DecodeBase64(TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "serialNumber"), decoded_stream_ser);
				}
				catch (std::runtime_error&)
				{
					std::cout << std::endl << "Error: Decoding Failed." << std::endl;
					return;
				}

				resultAll += "\t serialNumber=";
				for (std::vector<unsigned char>::iterator iter = decoded_stream_ser.begin(); iter != decoded_stream_ser.end(); ++iter)
				{
					ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
				}
				resultAll += ss.str() + "\n";
			}
			{
				std::vector<unsigned char> decoded_stream_key;
				std::stringstream ss;
				try
				{
					TSEUtil::DecodeBase64(TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "tsePublicKey"), decoded_stream_key);
				}
				catch (std::runtime_error&)
				{
					std::cout << std::endl << "Error: Decoding Failed." << std::endl;
					return;
				}

				resultAll += "\t tsePublicKey=";
				for (std::vector<unsigned char>::iterator iter = decoded_stream_key.begin(); iter != decoded_stream_key.end(); ++iter)
				{
					ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
				}
				resultAll += ss.str() + "\n";
			}
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}

void CuiOperateTSE::CuiGetStorageSmartInfo()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string smartInfo = "";
	std::string yORn = "n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET STORAGE SMART INFO]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << std::endl << "Get Smart Info in raw JSON string format (y/n): ";
	std::cin >> yORn;

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetStorageSmartInfo(smartInfo);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetStorageSmartInfo() : " + result + "\n";

	if (result == "EXECUTION_OK")
	{
		if (yORn == "y")
		{
			resultAll += "\t Raw JSON string=\n";
			resultAll += "\t " + smartInfo;
		}
		else
		{
			std::string data_integrity = TseSimpleJsonAndXmlParser::JsonGetObject(smartInfo, "dataIntegrity");
			std::string erase_lifetime_status = TseSimpleJsonAndXmlParser::JsonGetObject(smartInfo, "eraseLifetimeStatus");
			std::string spare_block_status = TseSimpleJsonAndXmlParser::JsonGetObject(smartInfo, "spareBlockStatus");

			resultAll += "\t [TSE SMART INFO]\n";
			resultAll += "\t dataIntegrity:\n";
			resultAll += "\t   healthStatus=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(data_integrity, "healthStatus") + "\"\n";
			resultAll += "\t   uncorrectableECCErrors=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(data_integrity, "uncorrectableECCErrors") + "\"\n";
			resultAll += "\t eraseLifetimeStatus:\n";
			resultAll += "\t   healthStatus=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(erase_lifetime_status, "healthStatus") + "\"\n";
			resultAll += "\t   remainingEraseCounts=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(erase_lifetime_status, "remainingEraseCounts") + "\"\n";
			resultAll += "\t spareBlockStatus:\n";
			resultAll += "\t   healthStatus=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(spare_block_status, "healthStatus") + "\"\n";
			resultAll += "\t   remainingSpareBlocks=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(spare_block_status, "remainingSpareBlocks") + "\"\n";
			resultAll += "\t remainingTenYearsDataRetention=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(smartInfo, "remainingTenYearsDataRetention") + "\"\n";
			resultAll += "\t isReplacementNeeded=\"" + TseSimpleJsonAndXmlParser::JsonGetNonStringValue(smartInfo, "isReplacementNeeded") + "\"\n";
			resultAll += "\t tseHealth=\"" + TseSimpleJsonAndXmlParser::JsonGetValue(smartInfo, "tseHealth") + "\"\n";			
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}

void CuiOperateTSE::CuiRunTseSelfTest()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[RUN TSE SELF TEST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{	
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->RunTSESelfTest();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t RunTSESelfTest() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
}

bool CuiOperateTSE::CuiFactoryReset() 
{
	std::string result = "";
	std::string result2 = "N/A";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string rcvTime2 = "N/A";
	std::string yORn = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[FACTORY RESET]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "WARNING!" << std::endl;
	std::cout << "Please note that this function exists only for development purposes and will not be available for the final TSEs." << std::endl;

	std::cout << std::endl << "Are you sure you want to reset the TSE (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->FactoryReset();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

		if (result == "EXECUTION_OK")
		{
			result2 = mTseOpr->RunTSESelfTest();
			rcvTime2 = std::to_string(mTseOpr->GetLastRcvTime());
		}

		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t FactoryReset() : " + result + "\n";
		resultAll += RCVTIMELABEL + rcvTime2 + "\t RunTSESelfTest() : " + result2 + "\n";
		if ((result2 == "EXECUTION_OK") || (result2 == "TSE1_ERROR_CLIENT_NOT_REGISTERED")) //just ignore the TSE1_ERROR_CLIENT_NOT_REGISTERED in this case.
		{
			resultAll += "Factory Reset has been successfully executed!";
		}
		else
		{
			resultAll += "Factory Reset is not successful.";
		}

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	return true;
}

void CuiOperateTSE::CuiSetup() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string puk = "";
	std::string adminPin = "";
	std::string timeAdminPin = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[SET UP]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "PUK: ";
	std::getline(std::cin, puk);

	std::cout << "Admin PIN: ";
	std::getline(std::cin, adminPin);

	std::cout << "Time Admin PIN: ";
	std::getline(std::cin, timeAdminPin);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->Setup(puk, adminPin, timeAdminPin);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t Setup() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		mIsTseInitialized = true;
	}
}

void CuiOperateTSE::CuiAdminLogin() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string pin = "";
	std::string secretKey = "";
	std::string challenge = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-IN AS ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "PIN: ";
	std::getline(std::cin, pin);
	std::cout << "Secret Key: ";
	std::getline(std::cin, secretKey);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetChallenge(mCurrentUserId, challenge);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

	resultAll += RCVTIMELABEL + rcvTime + "\t GetChallenge() : " + result + "\n";
	resultAll += "\t challenge=\"" + challenge + "\"\n";

	if (result == "EXECUTION_OK") 
	{
		std::string hash = "";
		std::string remainingRetries;

		std::string chaSctkeyS = challenge + secretKey;
		std::vector<unsigned char> chaSctkeyV (chaSctkeyS.begin(), chaSctkeyS.end());

		try
		{
			hash = TSEUtil::EncodeBase64Sha256(chaSctkeyV);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Encoding Failed." << std::endl;
			return;
		}

		result = mTseOpr->AuthenticateUserForAdmin(mCurrentUserId, pin, hash, remainingRetries);
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		resultAll += RCVTIMELABEL + rcvTime + "\t AuthenticateUserForAdmin() : " + result + "\n";
		if (result == "TSE1_ERROR_AUTHENTICATION_FAILED")
		{
			resultAll += "\t remainingRetries=\"" + remainingRetries + "\"\n";
		}
	}
	mTseOpr->TSECloseDevice();

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		mIsAdminLogin = true;
	}
}

void CuiOperateTSE::CuiAdminLogout() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-OUT AS ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to log-out (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->LogOutForAdmin();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t LogOutForAdmin() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if ((result == "EXECUTION_OK") || (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER"))
		{
			mIsAdminLogin = false;
		}
	}
}


void CuiOperateTSE::CuiAdminSetTimeout() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string timeoutAdmin = "";
	std::string timeoutTimeAdmin = "";
	std::string timeoutExport = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[SET TIMEOUT INTERVAL]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Note: Defaut timeouts are: 900 sec for Admin, 28800 sec for Time admin, 100 sec for Export." << std::endl;

	std::cout << "Please enter the following (in seconds):" << std::endl;
	getchar();
	while (true)
	{
		std::cout << "New Timeout for Admin: ";
		std::getline(std::cin, timeoutAdmin);
		if (TSEUtil::IsNumber(timeoutAdmin))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "New Timeout for TimeAdmin: ";
		std::getline(std::cin, timeoutTimeAdmin);
		if (TSEUtil::IsNumber(timeoutTimeAdmin))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "New Timeout for Export: ";
		std::getline(std::cin, timeoutExport);
		if (TSEUtil::IsNumber(timeoutExport))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->SetTimeOutInterval(timeoutAdmin, timeoutTimeAdmin, timeoutExport);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t SetTimeOutInterval() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}
}

void CuiOperateTSE::CuiAdminGetTimeout() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string timeoutAdmin = "";
	std::string timeoutTimeAdmin = "";
	std::string timeoutExport = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET TIMEOUT INTERVAL]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Note: Defaut timeouts are: 900 sec for Admin, 28800 sec for Time admin, 100 sec for Export." << std::endl;

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetTimeOutInterval(timeoutAdmin, timeoutTimeAdmin, timeoutExport);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetTimeOutInterval() : " + result + "\n";
	resultAll += "\t timeoutIntervalForAdmin=\"" + timeoutAdmin + "\"\n";
	resultAll += "\t timeoutIntervalForExport=\"" + timeoutExport + "\"\n";
	resultAll += "\t timeoutIntervalForTimeAdmin=\"" + timeoutTimeAdmin + "\"\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}
}

void CuiOperateTSE::CuiAdminSetUpForPrinter()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[SET UP FOR PRINTER]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->SetUpForPrinter();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t SetUpForPrinter() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
		{
			mIsAdminLogin = false;
		}
	}
}

void CuiOperateTSE::CuiUpdateTimeFirstTime() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	
	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TIME FOR FIRST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "This function is ignored if Data/Time has already been set-up." << std::endl;

	std::string utcTime = TSEUtil::GetCurrentUtcTime();
	mTseOpr->TSEOpenDevice();
	result = mTseOpr->UpdateTimeForFirst(mCurrentUserId, utcTime, "false");
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t UpdateTimeForFirst() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		std::cout << std::endl << "TIME UPDATED TO : " << utcTime << std::endl;
	}
	
	if ((result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER") || (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER"))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}	
	}
}

void CuiOperateTSE::CuiUpdateTime() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	
	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TIME]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::string utcTime = TSEUtil::GetCurrentUtcTime();
	mTseOpr->TSEOpenDevice();
	result = mTseOpr->UpdateTime(mCurrentUserId, utcTime, "false");
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t UpdateTime() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		std::cout << std::endl << "TIME UPDATED TO : " << utcTime << std::endl;
	}

	if ((result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER") || (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER"))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}
}

void CuiOperateTSE::CuiUpdateTimeManually()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	std::string newDate = "";
	std::string newTime = "";
	std::string newDateTime = "";
	std::string yORn = "";
	bool updateForFirst = false;

	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TIME MANUALLY (Manual)]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << std::endl << "UpdateTime for the first time? (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		updateForFirst = true;
	}

	getchar();
	while (true)
	{
		std::cout << "Enter the new Date (Format YYYY-MM-DD): ";
		std::getline(std::cin, newDate);
		std::string tmpDate = newDate + "T00:00:00Z";
		if (TSEUtil::IsDateTime(tmpDate))
		{
			break;
		}

		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	while (true)
	{
		std::cout << "Enter the new Time (Format HH:MM:SS): ";
		std::getline(std::cin, newTime);
		std::string tmpTime = "2020-01-01T" + newTime + "Z";
		if (TSEUtil::IsDateTime(tmpTime))
		{
			break;
		}

		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	newDateTime = newDate + "T" + newTime + "Z";
	mTseOpr->TSEOpenDevice();

	if (updateForFirst)
	{
		result = mTseOpr->UpdateTimeForFirst(mCurrentUserId, newDateTime, "false");
	}
	else
	{
		result = mTseOpr->UpdateTime(mCurrentUserId, newDateTime, "false");
	}
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	if (updateForFirst)
	{
		resultAll += RCVTIMELABEL + rcvTime + "\t UpdateTimeForFirst() : " + result + "\n";
	}
	else
	{
		resultAll += RCVTIMELABEL + rcvTime + "\t UpdateTime() : " + result + "\n";
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		std::cout << std::endl << "TIME UPDATED TO : " << newDateTime << std::endl;
	}

	if ((result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER") || (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER"))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}
}

void CuiOperateTSE::CuiAdminEnableExportIfCspTestFails()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[ENABLE EXPORT IF CSP TEST FAILS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->EnableExportIfCspTestFails();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t EnableExportIfCspTestFails() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
		{
			mIsAdminLogin = false;
		}
	}
}

void CuiOperateTSE::CuiAdminDisableExportIfCspTestFails()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[DISABLE EXPORT IF CSP TEST FAILS]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->DisableExportIfCspTestFails();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t DisableExportIfCspTestFails() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
		{
			mIsAdminLogin = false;
		}
	}
}

void CuiOperateTSE::CuiAdminGetExportDataArchive() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string exportResult = "";
	std::string exportData = "";
	std::string tarExportSize;
	std::vector<BYTE> exportDataAll;
	bool decodeIsComplete = true;

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - ARCHIVE EXPORT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->ArchiveExport(tarExportSize);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

	resultAll += RCVTIMELABEL + rcvTime + "\t ArchiveExport() : " + result + "\n";
	resultAll += "\t tarExportSize=\"" + tarExportSize + "\"\n";
	
	do
	{
		result = mTseOpr->GetExportData(exportData, exportResult);
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

		resultAll += RCVTIMELABEL + rcvTime + "\t GetExportData() : " + result + "\n";
		resultAll += "\t exportResult=\"" + exportResult + "\"\n";

		if (result != "EXECUTION_OK")
		{
			exportDataAll.clear();
			break;
		}

		if (exportResult == "EXPORT_INCOMPLETE_SEARCHING")
		{
			continue;
		}

		try 
		{
			TSEUtil::DecodeBase64(exportData, exportDataAll);
		}
		catch (std::runtime_error&)
		{
			std::cout << "Decoding export data failed." << std::endl;
			decodeIsComplete = false;
		}
	} while (exportResult != "EXPORT_COMPLETE");
	
	mTseOpr->TSECloseDevice();
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}

	if (decodeIsComplete && (exportDataAll.size() > 0))
	{
		std::string exportFilename = "";

		std::string tempDate = TSEUtil::GetCurrentUtcTime();
		tempDate.erase(std::remove(tempDate.begin(), tempDate.end(), ':'), tempDate.end());
		exportFilename = "Export_ARCHIVE_" + tempDate + std::string(EXPORTFILEEXT);
		remove(exportFilename.c_str());
		
		TSEUtil::SaveToFile(exportFilename, exportDataAll.data(), (unsigned long) exportDataAll.size());
		mLastExportFile = exportFilename;
		std::cout << "Export File Size=\"" << exportDataAll.size() << "\"" << std::endl;
		std::cout << "Export File: " << exportFilename << std::endl;
	}
}

void CuiOperateTSE::CuiAdminGetExportDataFilterByTrx() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	std::string exportResult = "";
	std::string exportData = "";
	std::string clientId;
	std::string trxNo;
	std::vector<BYTE> exportDataAll;
	bool decodeIsComplete = true;

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - EXPORT FILTERED BY TRANSACTION NUMBER]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	
	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, clientId);

	while (true)
	{
		std::cout << "Enter the Transaction No: ";
		std::getline(std::cin, trxNo);
		if (TSEUtil::IsNumber(trxNo))
		{
			break;
		}
			
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->ExportFilteredByTransactionNumber(clientId, trxNo);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

	resultAll += RCVTIMELABEL + rcvTime + "\t ExportFilteredByTransactionNumber() : " + result + "\n";

	do
	{
		result = mTseOpr->GetExportData(exportData, exportResult);
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

		resultAll += RCVTIMELABEL + rcvTime + "\t GetExportData() : " + result + "\n";
		resultAll += "\t exportResult=\"" + exportResult + "\"\n";

		if (result != "EXECUTION_OK")
		{
			exportDataAll.clear();
			break;
		}

		if (exportResult == "EXPORT_INCOMPLETE_SEARCHING")
		{
			continue;
		}

		try
		{
			TSEUtil::DecodeBase64(exportData, exportDataAll);
		}
		catch (std::runtime_error&)
		{
			std::cout << "Decoding export data failed." << std::endl;
			decodeIsComplete = false;
		}
	} while (exportResult != "EXPORT_COMPLETE");
	
	mTseOpr->TSECloseDevice();		

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}

	if (decodeIsComplete && (exportDataAll.size() > 0))
	{
		std::string exportFilename = "";

		exportFilename = "Export_" + clientId + "_TRXNUM_" + trxNo + EXPORTFILEEXT;
		remove(exportFilename.c_str());

		TSEUtil::SaveToFile(exportFilename, exportDataAll.data(), (unsigned long) exportDataAll.size());
		mLastExportFile = exportFilename;
		std::cout << "Export File Size=\"" << exportDataAll.size() << "\"" << std::endl;
		std::cout << "Export File: " << exportFilename << std::endl;
	}
}

void CuiOperateTSE::CuiAdminGetExportDataFilterByTrxNoInterval() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string exportResult = "";
	std::string exportData = "";
	std::string clientId = "";
	std::string startTrxNo = "";
	std::string endTrxNo = "";
	std::vector<BYTE> exportDataAll;
	bool decodeIsComplete = true;

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - EXPORT FILTERED BY TRANSACTION NUMBER INTERVAL]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, clientId);

	while (true)
	{
		std::cout << "Enter the Start Transaction No: ";
		std::getline(std::cin, startTrxNo);
		if (TSEUtil::IsNumber(startTrxNo))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "Enter the End Transaction No: ";
		std::getline(std::cin, endTrxNo);
		if (TSEUtil::IsNumber(endTrxNo))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->ExportFilteredByTransactionNumberInterval(clientId, startTrxNo, endTrxNo);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

	resultAll += RCVTIMELABEL + rcvTime + "\t ExportFilteredByTransactionNumberInterval() : " + result + "\n";

	do
	{
		result = mTseOpr->GetExportData(exportData, exportResult);
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

		resultAll += RCVTIMELABEL + rcvTime + "\t GetExportData() : " + result + "\n";
		resultAll += "\t exportResult=\"" + exportResult + "\"\n";

		if (result != "EXECUTION_OK")
		{
			exportDataAll.clear();
			break;
		}

		if (exportResult == "EXPORT_INCOMPLETE_SEARCHING")
		{
			continue;
		}

		try
		{
			TSEUtil::DecodeBase64(exportData, exportDataAll);
		}
		catch (std::runtime_error&)
		{
			std::cout << "Decoding export data failed." << std::endl;
			decodeIsComplete = false;
		}
	} while (exportResult != "EXPORT_COMPLETE");
	
	mTseOpr->TSECloseDevice();
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}

	if (decodeIsComplete && (exportDataAll.size() > 0))
	{
		std::string exportFilename = "";

		exportFilename = "Export_" + clientId + "_TRXRNG_" + startTrxNo + "_TO_" + endTrxNo + EXPORTFILEEXT;
		remove(exportFilename.c_str());

		TSEUtil::SaveToFile(exportFilename, exportDataAll.data(), (unsigned long) exportDataAll.size());
		mLastExportFile = exportFilename;
		std::cout << "Export File Size=\"" << exportDataAll.size() << "\"" << std::endl;
		std::cout << "Export File: " << exportFilename << std::endl;
	}
}

void CuiOperateTSE::CuiAdminGetExportDataFilterByPeriod() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	std::string exportResult = "";
	std::string exportData = "";
	std::string clientId = "";
	std::string startDate = "";
	std::string endDate = "";
	std::vector<BYTE> exportDataAll;
	bool decodeIsComplete = true;

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET EXPORT DATA - EXPORT FILTERED BY PERIOD OF TIME]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, clientId);
	
	while (true)
	{
		std::cout << "Enter the Start Date/Time (UTC Format YYYY-MM-DDTHH:MM:SSZ): ";
		std::getline(std::cin, startDate);
		if (TSEUtil::IsDateTime(startDate))
		{
			break;
		}
			
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}
	
	while (true)
	{
		std::cout << "Enter the End Date/Time (UTC Format YYYY-MM-DDTHH:MM:SSZ): ";
		std::getline(std::cin, endDate);
		if (TSEUtil::IsDateTime(endDate))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->ExportFilteredByPeriodOfTime(clientId, startDate, endDate);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	resultAll += RCVTIMELABEL + rcvTime + "\t ExportFilteredByPeriodOfTime() : " + result + "\n";

	do
	{
		result = mTseOpr->GetExportData(exportData, exportResult);
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

		resultAll += RCVTIMELABEL + rcvTime + "\t GetExportData() : " + result + "\n";
		resultAll += "\t exportResult=\"" + exportResult + "\"\n";

		if (result != "EXECUTION_OK")
		{
			exportDataAll.clear();
			break;
		}

		if (exportResult == "EXPORT_INCOMPLETE_SEARCHING")
		{
			continue;
		}

		try
		{
			TSEUtil::DecodeBase64(exportData, exportDataAll);
		}
		catch (std::runtime_error&)
		{
			std::cout << "Decoding export data failed." << std::endl;
			decodeIsComplete = false;
		}
	} while (exportResult != "EXPORT_COMPLETE");
	
	mTseOpr->TSECloseDevice();
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}

	if (decodeIsComplete && (exportDataAll.size() > 0))
	{
		std::string exportFilename = "";
		std::string tempSDate = startDate;
		std::string tempEDate = endDate;
		tempSDate.erase(std::remove(tempSDate.begin(), tempSDate.end(), ':'), tempSDate.end());
		tempEDate.erase(std::remove(tempEDate.begin(), tempEDate.end(), ':'), tempEDate.end());

		exportFilename = "Export_" + clientId + "_PERIOD_" + tempSDate + "_TO_" + tempEDate + EXPORTFILEEXT;
		remove(exportFilename.c_str());

		TSEUtil::SaveToFile(exportFilename, exportDataAll.data(), (unsigned long) exportDataAll.size());
		mLastExportFile = exportFilename;
		std::cout << "Export File Size=\"" << exportDataAll.size() << "\"" << std::endl;
		std::cout << "Export File: " << exportFilename << std::endl;
	}
}

void CuiOperateTSE::CuiAdminFinalizeExport() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	std::string yORn = "";
	std::string deleteData;

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[FINALIZE EXPORT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << std::endl << "Do you want to delete the exported data from the TSE (y/n): ";
	std::cin >> yORn;
	deleteData = (yORn == "y") ? "true" : "false";

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->FinalizeExport(deleteData);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t FinalizeExport() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}
}

void CuiOperateTSE::CuiAdminCancelExportData() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	std::string yORn = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[CANCEL EXPORT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->CancelExport();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t CancelExport() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (mLastExportFile != "")
		{
			remove(mLastExportFile.c_str());
		}

		if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
		{
			mIsAdminLogin = false;
		}
	}
}

void CuiOperateTSE::CuiAdminRegisterSecretKey() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	std::string secretKey = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[SET-UP SHARED SECRET KEY]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "Secret Key: ";
	std::getline(std::cin, secretKey);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->RegisterSecretKey(secretKey);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t RegisterSecretKey() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}
}

void CuiOperateTSE::CuiAdminDisableSecureElement() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[DISABLE SECURE ELEMENT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->DisableSecureElement();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t DisableSecureElement() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
		{
			mIsAdminLogin = false;
		}
	}
}

void CuiOperateTSE::CuiChangeUserPin() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	std::string oldPin = "";
	std::string newPin = "";
	std::string remainingRetries = "";

	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		if (IsAdminLogin() == false)
		{
			return;
		}
	}
	else
	{
		if (IsTimeAdminLogin() == false)
		{
			return;
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[CHANGE PIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "Old PIN: ";
	std::getline(std::cin, oldPin);
	std::cout << "New PIN: ";
	std::getline(std::cin, newPin);

	mTseOpr->TSEOpenDevice();
	if (mCurrentUserId == EPSON_TSE_ADMIN)
	{
		result = mTseOpr->ChangePinForAdmin(mCurrentUserId, oldPin, newPin, remainingRetries);
	}
	else
	{
		result = mTseOpr->ChangePinForTimeAdmin(mCurrentUserId, oldPin, newPin, remainingRetries);
	}
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t ChangePinForAdmin() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if ((result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER") || (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER"))
	{
		if (mCurrentUserId == EPSON_TSE_ADMIN)
		{
			mIsAdminLogin = false;
		}
		else
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}
}

void CuiOperateTSE::CuiAdminRegisterClient() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string clientId = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[REGISTER CLIENT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please set-up the following:" << std::endl;
	getchar();
	std::cout << "New Client Id: ";
	std::getline(std::cin, clientId);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->RegisterClient(clientId);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t RegisterClient() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}
}

void CuiOperateTSE::CuiAdminDeregisterClient() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string clientId = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[DEREGISTER CLIENT]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please enter the client id to remove:" << std::endl;
	getchar();
	std::cout << "Client Id: ";
	std::getline(std::cin, clientId);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->DeregisterClient(clientId);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t DeregisterClient() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}
}

void CuiOperateTSE::CuiAdminUnlockTse() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UNLOCK TSE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to continue (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->UnlockTSE();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t UnlockTSE() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
		{
			mIsAdminLogin = false;
		}
	}

}

void CuiOperateTSE::CuiAdminLockTse() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string yORn = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOCK TSE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Are you sure you want to continue? (y/n) : ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->LockTSE();
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t LockTSE() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
		{
			mIsAdminLogin = false;
		}
	}
}


void CuiOperateTSE::CuiAdminGetAuthenticatedUserList() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string selOpt = "";
	int nOpt = -1;
	std::string list = "";
	std::string userRole = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	while (true)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[GET AUTHENTICATED USER LIST]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		std::cout << "Select the number that correspond to the User Role you want to check:" << std::endl;
		std::cout << "[1] Admin " << std::endl;
		std::cout << "[2] TimeAdmin " << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> selOpt;

		if (TSEUtil::IsNumber(selOpt))
		{
			std::stringstream ssTmp(selOpt);
			ssTmp >> nOpt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}

		if (nOpt == 1)
		{
			userRole = "Admin";
			break;
		}
		else if (nOpt == 2)
		{
			userRole = "TimeAdmin";
			break;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}
	}

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetAuthenticatedUserList(userRole, list);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetAuthenticatedUserList() : " + result + "\n";
	resultAll += "\t authenticatedUserList=";

	std::vector<std::string> tokens;
	LStringToVector(list, tokens);
	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{
		std::string tok = *str_iter;

		size_t index = 0;
		while (true)
		{
			index = tok.find("\\/", index);
			if (index == std::string::npos) 
				break;
			tok.replace(index, 2, "/");
			index += 1;
		}
		
		resultAll += "\n\t\t" + tok + ", ";
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_ADMIN_USER")
	{
		mIsAdminLogin = false;
	}
}

void CuiOperateTSE::CuiGetLogMsgCertificate() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string cert = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET LOG MESSAGE CERTIFICATE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetLogMessageCertificate(cert);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetLogMessageCertificate() : " + result + "\n";
	resultAll += "logMessageCertificate=\n";

	std::vector<unsigned char> decoded_stream;
	try
	{
		TSEUtil::DecodeBase64(cert, decoded_stream);
	}
	catch (std::runtime_error&) {
		std::cout << std::endl << "Error: Decoding Failed." << std::endl;
		return;
	}
	resultAll += std::string(decoded_stream.begin(), decoded_stream.end());

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}

void CuiOperateTSE::CuiGetLastTrxResponse() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string clientId = "";
	std::string rcvLogTime = "";
	std::string rcvTrxNo = "";
	std::string rcvSigCntr = "";
	std::string rcvSerialNo = "";
	std::string rcvSignature = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET LAST TRANSACTION RESPONSE]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, clientId);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetLastTransactionResponse(clientId, rcvTrxNo, rcvLogTime, rcvSigCntr, rcvSignature, rcvSerialNo);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetLastTransactionResponse() : " + result + "\n";
	if (result == "EXECUTION_OK")
	{
		resultAll += "\t transactionNumber=\"" + rcvTrxNo + "\"\n";
		resultAll += "\t logTime=\"" + rcvLogTime + "\"\n";
		resultAll += "\t signatureCounter=\"" + rcvSigCntr + "\"\n";

		{
			std::vector<unsigned char> decoded_stream_ser;
			std::stringstream ss;
			try
			{
				TSEUtil::DecodeBase64(rcvSerialNo, decoded_stream_ser);
			}
			catch (std::runtime_error&)
			{
				std::cout << std::endl << "Error: Decoding Failed." << std::endl;
				return;
			}

			resultAll += "\t serialNumber=";
			for (std::vector<unsigned char>::iterator iter = decoded_stream_ser.begin(); iter != decoded_stream_ser.end(); ++iter)
			{
				ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
			}
			resultAll += ss.str() + "\n";
		}
		{
			std::vector<unsigned char> decoded_stream_sig;
			std::stringstream ss;
			try
			{
				TSEUtil::DecodeBase64(rcvSignature, decoded_stream_sig);
			}
			catch (std::runtime_error&)
			{
				std::cout << std::endl << "Error: Decoding Failed." << std::endl;
				return;
			}

			resultAll += "\t signature=";
			for (std::vector<unsigned char>::iterator iter = decoded_stream_sig.begin(); iter != decoded_stream_sig.end(); ++iter)
			{
				ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
			}
			resultAll += ss.str() + "\n";
		}
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}

void CuiOperateTSE::CuiGetStartedTrxList()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string clientId = "";
	std::string list = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET STARTED TRANSACTION LIST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	getchar();
	std::cout << "Enter the Client Id: ";
	std::getline(std::cin, clientId);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetStartedTransactionList(clientId, list);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetStartedTransactionList() : " + result + "\n";
	resultAll += "\t startedTransactionNumberList=\"" + list + "\"\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}

void CuiOperateTSE::CuiExportSerialNumber()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string tseInfo = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET SERIAL NUMBER]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetStorageInfo(tseInfo);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetStorageInfo() : " + result + "\n";
	{
		std::vector<unsigned char> decoded_stream_ser;
		std::stringstream ss;
		try
		{
			TSEUtil::DecodeBase64(TseSimpleJsonAndXmlParser::JsonGetValue(tseInfo, "serialNumber"), decoded_stream_ser);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Decoding Failed." << std::endl;
			return;
		}

		resultAll += "\t serialNumber=";
		for (std::vector<unsigned char>::iterator iter = decoded_stream_ser.begin(); iter != decoded_stream_ser.end(); ++iter)
		{
			ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
		}
		resultAll += ss.str() + "\n";
	}
	
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

}

void CuiOperateTSE::CuiAdminGetRegisteredClientList() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string list = "";

	if (IsAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[GET REGISTERED CLIENT ID LIST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetRegisteredClientList(list);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t GetRegisteredClientList() : " + result + "\n";
	resultAll += "\t registeredClientIdList=";

	std::vector<std::string> tokens;
	LStringToVector(list, tokens);
	for (std::vector<std::string>::iterator str_iter = tokens.begin(); str_iter != tokens.end(); ++str_iter)
	{		
		std::string tok = *str_iter;

		size_t index = 0;
		while (true)
		{
			index = tok.find("\\/", index);
			if (index == std::string::npos) 
				break;
			tok.replace(index, 2, "/");
			index += 1;
		}
		
		resultAll += "\n\t\t" + tok + ", ";

	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}

void CuiOperateTSE::CuiTimeAdmin()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string selOpt = "";
	int nOpt = -1;

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[OPERATE AS TIME ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please enter the Client Id." << std::endl;
	std::cout << "Client Id: ";
	std::cin >> mCurrentUserId;

	while (true)
	{
		bool isTimeAdminLogin = IsTimeAdminLogin();
		
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[OPERATE AS TIME ADMIN]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
		
		std::cout << "Client ID: " << mCurrentUserId << std::endl;
		std::cout << "LOGGED-IN? : " << (isTimeAdminLogin ? "Yes" : "No") << std::endl << std::endl;
		
		std::cout << "Please select the operation you want to perform" << std::endl;

		if (isTimeAdminLogin)
		{
			std::cout << "[1] Log-out" << std::endl;
			std::cout << "[2] Update Date/Time for the first time" << std::endl;
			std::cout << "[3] Update Date/Time" << std::endl;
			std::cout << "[4] Start Transaction" << std::endl;
			std::cout << "[5] Update Transaction" << std::endl;
			std::cout << "[6] Finish Transaction" << std::endl;
			std::cout << "[7] Change PIN For TimeAdmin" << std::endl;
		}
		else
		{
			std::cout << "[1] Log-in" << std::endl;
		}
		std::cout << "[99] Exit Menu" << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> selOpt;

		if (TSEUtil::IsNumber(selOpt))
		{
			std::stringstream ssTmp(selOpt);
			ssTmp >> nOpt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}
			
		if (nOpt == 99)
		{
			break;
		}
		try
		{
			switch (nOpt)
			{
				case 1:
				{
					if (isTimeAdminLogin)
					{
						CuiTimeAdminLogout();
					}
					else
					{
						CuiTimeAdminLogin();
					}
					break;
				}
				case 2: CuiUpdateTimeFirstTime();  break;
				case 3: CuiUpdateTime();  break;
				case 4: CuiTimeAdminStartTransaction(); break;
				case 5: CuiTimeAdminUpdateTransaction(); break;
				case 6: CuiTimeAdminFinishTransaction(); break;
				case 7: CuiChangeUserPin();  break;
				case 99: break;
				default:
				{
					SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
					std::cout << "INVALID INPUT!" << std::endl;
					SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
					continue;
				}
			}
		}
		catch (std::runtime_error&) 
		{
			std::cout << "Error while executing option [" << nOpt << "]:" << std::endl;
		}//try catch
	}//while
}

void CuiOperateTSE::CuiTimeAdminLogin() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string pin = "";
	std::string secretKey = "";
	std::string challenge = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-IN AS TIME ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "PIN: ";
	std::getline(std::cin, pin);
	std::cout << "Secret Key: ";
	std::getline(std::cin, secretKey);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetChallenge(mCurrentUserId, challenge);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

	resultAll += RCVTIMELABEL + rcvTime + "\t GetChallenge() : " + result + "\n";
	resultAll += "\t challenge=\"" + challenge + "\"\n";

	if (result == "EXECUTION_OK")
	{
		std::string hash = "";
		std::string remainingRetries;

		std::string chaSctkeyS = challenge + secretKey;
		std::vector<unsigned char> chaSctkeyV(chaSctkeyS.begin(), chaSctkeyS.end());

		try
		{
			hash = TSEUtil::EncodeBase64Sha256(chaSctkeyV);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Encoding Failed." << std::endl;
			return;
		}

		result = mTseOpr->AuthenticateUserForTimeAdmin(mCurrentUserId, pin, hash, remainingRetries);
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		resultAll += RCVTIMELABEL + rcvTime + "\t AuthenticateUserForAdmin() : " + result + "\n";
		if (result == "TSE1_ERROR_AUTHENTICATION_FAILED")
		{
			resultAll += "\t remainingRetries=\"" + remainingRetries + "\"\n";
		}
	}
	mTseOpr->TSECloseDevice();

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		mClientIdLogInList.push_back(mCurrentUserId);
	}
}

void CuiOperateTSE::CuiTimeAdminLogout() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string yORn = "";

	if (IsTimeAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[LOG-OUT AS TIME ADMIN]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << std::endl << "Are you sure you want to log-out (y/n): ";
	std::cin >> yORn;

	if (yORn == "y")
	{
		std::string rcvTime = "";
		
		mTseOpr->TSEOpenDevice();
		result = mTseOpr->LogOutForTimeAdmin(mCurrentUserId);
		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		mTseOpr->TSECloseDevice();

		resultAll += RCVTIMELABEL + rcvTime + "\t LogOutForTimeAdmin() : " + result + "\n";

		SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
		std::cout << std::endl << "RESULT : " << std::endl;
		std::cout << resultAll << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		if ((result == "EXECUTION_OK") || (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER"))
		{
			mClientIdLogInList.remove(mCurrentUserId);
		}
	}
}


void CuiOperateTSE::CuiTimeAdminStartTransaction() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string processType = "";
	std::string processData = "";
	std::string addtnlData = "";
	std::string rcvTrxNo = "";
	std::string rcvSigCntr = "";
	std::string rcvLogTime = "";
	std::string rcvSerialNo = "";
	std::string rcvSignature = "";

	std::vector<unsigned char> processDataV;
	std::vector<unsigned char> addtnlDataV;

	if (IsTimeAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[START TRANSACTION]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	std::cout << "Type: ";
	std::getline(std::cin, processType);

	while (true)
	{
		std::cout << "Process Data: ";
		std::getline(std::cin, processData);
		processDataV.clear();
		if ((processData.length() >= 3) && (processData.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(processData.substr(3), processDataV);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				break;
			}
		}
		else
		{
			std::copy(processData.begin(), processData.end(), std::back_inserter(processDataV));
			break;
		}
	}//while

	while (true)
	{
		std::cout << "Additional Data: ";
		std::getline(std::cin, addtnlData);
		addtnlDataV.clear();
		if ((addtnlData.length() >= 3) && (addtnlData.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(addtnlData.substr(3), addtnlDataV);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				break;
			}
		}
		else
		{
			std::copy(addtnlData.begin(), addtnlData.end(), std::back_inserter(addtnlDataV));
			break;
		}
	}//while

	std::string encProData = ""; 
	std::string encAddData = "";

	try
	{
		if (processData.size() != 0)
		{
			encProData = TSEUtil::EncodeBase64(processDataV);
		}
		if (addtnlData.size() != 0)
		{
			encAddData = TSEUtil::EncodeBase64(addtnlDataV);
		}
	}
	catch (std::runtime_error&)
	{
		std::cout << std::endl << "Error: Encoding Failed." << std::endl;
		return;
	}

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->StartTransaction(mCurrentUserId,
		encProData,
		processType,
		encAddData,
		rcvTrxNo,
		rcvLogTime,
		rcvSerialNo,
		rcvSigCntr,
		rcvSignature);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t StartTransaction() : " + result + "\n";
	resultAll += "\t transactionNumber=\"" + rcvTrxNo + "\"\n";
	resultAll += "\t logTime=\"" + rcvLogTime + "\"\n";
	resultAll += "\t signatureCounter=\"" + rcvSigCntr + "\"\n";

	{
		std::vector<unsigned char> decoded_stream_ser;
		std::stringstream ss;
		try
		{
			TSEUtil::DecodeBase64(rcvSerialNo, decoded_stream_ser);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Decoding Failed." << std::endl;
			return;
		}

		resultAll += "\t serialNumber=";
		for (std::vector<unsigned char>::iterator iter = decoded_stream_ser.begin(); iter != decoded_stream_ser.end(); ++iter)
		{
			ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
		}
		resultAll += ss.str() + "\n";
	}
	{
		std::vector<unsigned char> decoded_stream_sig;
		std::stringstream ss;
		try
		{
			TSEUtil::DecodeBase64(rcvSignature, decoded_stream_sig);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Decoding Failed." << std::endl;
			return;
		}

		resultAll += "\t signature=";
		for (std::vector<unsigned char>::iterator iter = decoded_stream_sig.begin(); iter != decoded_stream_sig.end(); ++iter)
		{
			ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
		}
		resultAll += ss.str() + "\n";
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER")
	{
		mClientIdLogInList.remove(mCurrentUserId);
	}
}

void CuiOperateTSE::CuiTimeAdminUpdateTransaction()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string processType = "";
	std::string processData = "";
	std::string addtnlData = "";
	std::string trxNo = "";
	std::string rcvSigCntr = "";
	std::string rcvLogTime = "";
	std::string rcvSignature = "";

	std::vector<unsigned char> processDataV;

	if (IsTimeAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UPDATE TRANSACTION]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	getchar();
	while (true)
	{
		std::cout << "Transaction No: ";
		std::getline(std::cin, trxNo);
		if (TSEUtil::IsNumber(trxNo))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	std::cout << "Type: ";
	std::getline(std::cin, processType);

	while (true)
	{
		std::cout << "Process Data: ";
		std::getline(std::cin, processData);
		processDataV.clear();
		if ((processData.length() >= 3) && (processData.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(processData.substr(3), processDataV);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				break;
			}
		}
		else
		{
			std::copy(processData.begin(), processData.end(), std::back_inserter(processDataV));
			break;
		}
	}//while

	std::string encProData = ""; 

	try
	{
		if (processData.size() != 0)
		{
			encProData = TSEUtil::EncodeBase64(processDataV);
		}
	}
	catch (std::runtime_error&)
	{
		std::cout << std::endl << "Error: Encoding Failed." << std::endl;
		return;
	}
	
	mTseOpr->TSEOpenDevice();
	result = mTseOpr->UpdateTransaction(mCurrentUserId,
		trxNo,
		encProData,
		processType,
		rcvLogTime,
		rcvSigCntr,
		rcvSignature);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t UpdateTransaction() : " + result + "\n";
	resultAll += "\t [IN]transactionNumber=\"" + trxNo + "\"\n";
	resultAll += "\t logTime=\"" + rcvLogTime + "\"\n";
	resultAll += "\t signatureCounter=\"" + rcvSigCntr + "\"\n";
	{
		std::vector<unsigned char> decoded_stream_sig;
		std::stringstream ss;
		try
		{
			TSEUtil::DecodeBase64(rcvSignature, decoded_stream_sig);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Decoding Failed." << std::endl;
			return;
		}

		resultAll += "\t signature=";
		for (std::vector<unsigned char>::iterator iter = decoded_stream_sig.begin(); iter != decoded_stream_sig.end(); ++iter)
		{
			ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
		}
		resultAll += ss.str() + "\n";
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	
	if (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER")
	{
		mClientIdLogInList.remove(mCurrentUserId);
	}
}

void CuiOperateTSE::CuiTimeAdminFinishTransaction() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string processType = "";
	std::string processData = "";
	std::string addtnlData = "";
	std::string trxNo = "";
	std::string rcvSigCntr = "";
	std::string rcvLogTime = "";
	std::string rcvSignature = "";

	std::vector<unsigned char> processDataV;
	std::vector<unsigned char> addtnlDataV;

	if (IsTimeAdminLogin() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[FINISH TRANSACTION]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	getchar();
	while (true)
	{
		std::cout << "Transaction No: ";
		std::getline(std::cin, trxNo);
		if (TSEUtil::IsNumber(trxNo))
		{
			break;
		}
		SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
		std::cout << "INVALID INPUT!" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	}

	std::cout << "Type: ";
	std::getline(std::cin, processType);

	while (true)
	{
		std::cout << "Process Data: ";
		std::getline(std::cin, processData);
		processDataV.clear();
		if ((processData.length() >= 3) && (processData.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(processData.substr(3), processDataV);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				break;
			}
		}
		else
		{
			std::copy(processData.begin(), processData.end(), std::back_inserter(processDataV));
			break;
		}
	}//while

	while (true)
	{
		std::cout << "Additional Data: ";
		std::getline(std::cin, addtnlData);
		addtnlDataV.clear();
		if ((addtnlData.length() >= 3) && (addtnlData.substr(0, 3) == "&0x")) //hexstring (binary data)
		{
			bool ret = GetHexString(addtnlData.substr(3), addtnlDataV);
			if (ret == false)
			{
				SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
				std::cout << "INVALID INPUT!" << std::endl;
				SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
				continue;
			}
			else
			{
				break;
			}
		}
		else
		{
			std::copy(addtnlData.begin(), addtnlData.end(), std::back_inserter(addtnlDataV));
			break;
		}
	}//while

	std::string encProData = ""; 
	std::string encAddData = "";

	try
	{
		if (processData.size() != 0)
		{
			encProData = TSEUtil::EncodeBase64(processDataV);
		}
		if (addtnlData.size() != 0)
		{
			encAddData = TSEUtil::EncodeBase64(addtnlDataV);
		}
	}
	catch (std::runtime_error&)
	{
		std::cout << std::endl << "Error: Encoding Failed." << std::endl;
		return;
	}

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->FinishTransaction(mCurrentUserId,
		trxNo,
		encProData,
		processType,
		encAddData,
		rcvLogTime,
		rcvSigCntr,
		rcvSignature);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t FinishTransaction() : " + result + "\n";
	resultAll += "\t [IN]transactionNumber=\"" + trxNo + "\"\n";
	resultAll += "\t logTime=\"" + rcvLogTime + "\"\n";
	resultAll += "\t signatureCounter=\"" + rcvSigCntr + "\"\n";
	{
		std::vector<unsigned char> decoded_stream_sig;
		std::stringstream ss;
		try
		{
			TSEUtil::DecodeBase64(rcvSignature, decoded_stream_sig);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Decoding Failed." << std::endl;
			return;
		}

		resultAll += "\t signature=";
		for (std::vector<unsigned char>::iterator iter = decoded_stream_sig.begin(); iter != decoded_stream_sig.end(); ++iter)
		{
			ss << std::setfill('0') << std::setw(2) << std::hex << (unsigned int)*iter << " ";
		}
		resultAll += ss.str() + "\n";
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "OTHER_ERROR_UNAUTHENTICATED_TIME_ADMIN_USER")
	{
		mClientIdLogInList.remove(mCurrentUserId);
	}
}

void CuiOperateTSE::CuiHost()
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string selOpt = "";
	int nOpt = -1;

	mCurrentUserId = EPSON_TSE_ADMIN;

	while (true)
	{
		SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
		std::cout << std::endl << "[OPERATE AS HOST]" << std::endl;
		SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

		std::cout << "Authenticated? : " << (mIsHostAuthenticated ? "Yes" : "No") << std::endl << std::endl;

		std::cout << "Please select the operation you want to perform" << std::endl;
		if (mIsHostAuthenticated)
		{
			std::cout << "[1] Deauthenticate Host" << std::endl;
			std::cout << "[2] Unblock User for Admin" << std::endl;
			std::cout << "[3] Unblock User for TimeAdmin" << std::endl;
			std::cout << "[4] Change PUK" << std::endl;
		}
		else
		{
			std::cout << "[1] Authenticate Host" << std::endl;
		}
		std::cout << "[99] Exit Menu" << std::endl;
		std::cout << std::endl << "Enter only the number: ";
		std::cin >> selOpt;

		if (TSEUtil::IsNumber(selOpt))
		{
			std::stringstream ssTmp(selOpt);
			ssTmp >> nOpt;
		}
		else
		{
			SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
			std::cout << "INVALID INPUT!" << std::endl;
			SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
			continue;
		}

		if (nOpt == 99)
		{
			break;
		}
		try
		{
			switch (nOpt)
			{
				case 1:
				{
					if (mIsHostAuthenticated) CuiHostDeauthenticate();
					else CuiHostAuthenticate();
					break;
				}
				case 2: CuiHostUnblockUser(true);  break;
				case 3: CuiHostUnblockUser(false);  break;
				case 4: CuiHostChangePuk();  break;
				case 99: break;
				default:
				{
					SetConsoleTextAttribute(mConsole, MENU_COLOR_WARNING);
					std::cout << "INVALID INPUT!" << std::endl;
					SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
					continue;
				}
			}
		}
		catch (std::runtime_error&)
		{
			std::cout << "Error while executing option [" << nOpt << "]:" << std::endl;
		}//try catch
	}
}

void CuiOperateTSE::CuiHostAuthenticate() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string secretKey = "";
	std::string challenge = "";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[AUTHENTICATE HOST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "Secret Key: ";
	std::getline(std::cin, secretKey);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->GetChallenge(mCurrentUserId, challenge);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());

	resultAll += RCVTIMELABEL + rcvTime + "\t GetChallenge() : " + result + "\n";
	resultAll += "\t challenge=\"" + challenge + "\"\n";

	if (result == "EXECUTION_OK")
	{
		std::string hash = "";

		std::string chaSctkeyS = challenge + secretKey;
		std::vector<unsigned char> chaSctkeyV(chaSctkeyS.begin(), chaSctkeyS.end());

		try
		{
			hash = TSEUtil::EncodeBase64Sha256(chaSctkeyV);
		}
		catch (std::runtime_error&)
		{
			std::cout << std::endl << "Error: Encoding Failed." << std::endl;
			return;
		}
		result = mTseOpr->AuthenticateHost(mCurrentUserId, hash);

		rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
		resultAll += RCVTIMELABEL + rcvTime + "\t AuthenticateHost() : " + result + "\n";

	}
	mTseOpr->TSECloseDevice();

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		mIsHostAuthenticated = true;
	}
}

void CuiOperateTSE::CuiHostDeauthenticate() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";

	if (IsHostAuthenticated() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[DEAUTHENTICATE HOST]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->DeauthenticateHost(mCurrentUserId);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t DeauthenticateHost() : " + result + "\n";

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);

	if (result == "EXECUTION_OK")
	{
		mIsHostAuthenticated = false;
	}
}

void CuiOperateTSE::CuiHostUnblockUser(bool tIsAdmin)
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string puk = "";
	std::string pin = "";
	std::string remainingRetries = "";

	if (IsHostAuthenticated() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[UNBLOCK USER]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "PUK: ";
	std::getline(std::cin, puk);
	std::cout << "New PIN: ";
	std::getline(std::cin, pin);

	mTseOpr->TSEOpenDevice();
	if (tIsAdmin)
	{
		result = mTseOpr->UnblockUserForAdmin(EPSON_TSE_ADMIN, puk, pin, remainingRetries);
	}
	else
	{
		result = mTseOpr->UnblockUserForTimeAdmin(EPSON_TSE_ADMIN, puk, pin, remainingRetries);
	}

	
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	if (tIsAdmin)
	{
		resultAll += RCVTIMELABEL + rcvTime + "\t UnblockUserForAdmin() : " + result + "\n";
	}
	else
	{
		resultAll += RCVTIMELABEL + rcvTime + "\t UnblockUserForTimeAdmin() : " + result + "\n";
	}
	
	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}

void CuiOperateTSE::CuiHostChangePuk() 
{
	std::string result = "";
	std::string resultAll = "";
	std::string rcvTime = "";
	std::string oldPuk = "";
	std::string newPuk = "";
	std::string remainingRetries = "";

	if (IsHostAuthenticated() == false)
	{
		return;
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_HEADER);
	std::cout << std::endl << "[CHANGE PUK]" << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
	std::cout << "Please enter the following:" << std::endl;
	getchar();
	std::cout << "Old PUK: ";
	std::getline(std::cin, oldPuk);
	std::cout << "New PUK: ";
	std::getline(std::cin, newPuk);

	mTseOpr->TSEOpenDevice();
	result = mTseOpr->ChangePuk(mCurrentUserId, oldPuk, newPuk, remainingRetries);
	rcvTime = std::to_string(mTseOpr->GetLastRcvTime());
	mTseOpr->TSECloseDevice();

	resultAll += RCVTIMELABEL + rcvTime + "\t ChangePuk() : " + result + "\n";
	if (result == "TSE1_ERROR_AUTHENTICATION_FAILED")
	{
		resultAll += "\t remainingRetries=\"" + remainingRetries + "\"\n";
	}

	SetConsoleTextAttribute(mConsole, MENU_COLOR_RESULT);
	std::cout << std::endl << "RESULT : " << std::endl;
	std::cout << resultAll << std::endl;
	SetConsoleTextAttribute(mConsole, MENU_COLOR_DEFAULT);
}


bool CuiOperateTSE::IsAdminLogin()
{
	return mIsAdminLogin;
}

bool CuiOperateTSE::IsTimeAdminLogin()
{
	auto it = std::find(mClientIdLogInList.begin(), mClientIdLogInList.end(), mCurrentUserId);
	return (it != mClientIdLogInList.end());
}

bool CuiOperateTSE::IsHostAuthenticated()
{
	return mIsHostAuthenticated;
}

bool CuiOperateTSE::GetHexString(std::string hexString, std::vector<unsigned char> &binData)
{
	binData.clear();

	if ((hexString.length() == 0) || ((hexString.length() % 2) != 0))
	{
		return false;
	}
	for (unsigned long i = 0; (i + 1) < hexString.length(); i += 2)
	{
		const char chr1 = hexString.at(i);
		const char chr2 = hexString.at(i + 1);

		unsigned char b1;
		unsigned char b2;

		unsigned char byteS = 0;

		if (chr1 >= '0' && chr1 <= '9')
		{
			b1 = chr1 - '0';
		}
		else if (chr1 >= 'a' && chr1 <= 'f')
		{
			b1 = chr1 - 'a' + 10;
		}
		else if (chr1 >= 'A' && chr1 <= 'F')
		{
			b1 = chr1 - 'A' + 10;
		}
		else
		{
			return false;
		}

		if (chr2 >= '0' && chr2 <= '9')
		{
			b2 = chr2 - '0';
		}
		else if (chr2 >= 'a' && chr2 <= 'f')
		{
			b2 = chr2 - 'a' + 10;
		}
		else if (chr2 >= 'A' && chr2 <= 'F')
		{
			b2 = chr2 - 'A' + 10;
		}
		else
		{
			return false;
		}

		byteS = (b1 * 16) + b2;
		binData.push_back(byteS);
	}
	return true;
}



void CuiOperateTSE::LStringToVector(const std::string& tString, std::vector<std::string> &tStr)
{

	std::regex rgx(",(?=(?:[^\"]*\"[^\"]*\")*[^\"]*$)");
	std::sregex_token_iterator reg_iter(tString.begin(), tString.end(), rgx, -1);
	std::sregex_token_iterator end;

	tStr.clear();
	if (tString == "")
		return;

	for (; reg_iter != end; ++reg_iter)
	{
		std::string match = reg_iter->str();
		match = match.substr(1, match.size() - 2);
		tStr.push_back(match);
	}
}

//EOF