// Copyright 2019-2020 Seiko Epson Corporation.
// Epson Europe
// EEB/RDC
// 2020/04  ka
// This Application is intended for demonstration purposes only and not for production environment.

#include "stdafx.h"
#include <iostream>
#include <sstream>
#include <string>
#include "../../Core/AccessMethod.h"
#include "../../Core/TSEOptions.h"
#include "../../Core/TSEUtil.h"
#include "../../Core/TSEPrintLog.h"
#include "CuiOperateTSE.h"
#include "TSELogOptions.h"
#include "EpsonTseDemo.h"

void PrintUsage(char * TExeName)
{
	std::cout << "Usage:" << std::endl;
	std::cout << "      " << TExeName << "  <required> <optional>" << std::endl;
	std::cout << std::endl;
	std::cout << "[Required]" << std::endl;
	std::cout << "  -mX: X=<Access Method>. Default is -m" << ACCESS_BY_SERIAL_ESCPOS << "." << std::endl;
	std::cout << "    " << ACCESS_BY_SERIAL_ESCPOS << " :ACCESS_BY_SERIAL_ESCPOS" << std::endl;
	std::cout << "    " << ACCESS_BY_NETWORK_ESCPOS << " :ACCESS_BY_NETWORK_ESCPOS" << std::endl;
	std::cout << "    " << ACCESS_BY_EPOS_DEVICE << " :ACCESS_BY_EPOS_DEVICE_XML" << std::endl;
	std::cout << "  -iX: X=<Port Identifier> (COM port/IP Address) e.g. -iCOM1, -i192.168.1.132" << std::endl;
	std::cout << std::endl;
	std::cout << "[Optional]" << std::endl;
	std::cout << "  -tX: X=<TSE Handle>. Default is \"local_TSE\"" << std::endl;
	std::cout << "[Optional - for setting logs]" << std::endl;
	std::cout << "  -wX: X=<Log level>." << std::endl;
	std::cout << "    0: Disabled, 1: Error, 2: Warning, 3: Info," << std::endl;
	std::cout << "    4: Trace, 5: Verbose" << std::endl;
	std::cout << "  -x1: Enable logging on DebugViewer." << std::endl;
	std::cout << "  -y1: Enable logging on File." << std::endl;
	std::cout << "  -z1: Enable logging on Standard Output." << std::endl;
	std::cout << std::endl;
	std::cout << "e.g.: " << TExeName << "  -m0   -iCOM1	           " << std::endl;
	std::cout << "      " << TExeName << "  -m1   -i192.168.192.168    " << std::endl;
	std::cout << "      " << TExeName << "  -m2   -i192.168.192.168    -tTSE1_xxxx" << std::endl;
	std::cout << "[To get the list of TSE devices connected to a server]" << std::endl;
	std::cout << "      " << TExeName << "  -m2   -i<server's ip address>" << std::endl;
}

void ShowOptions(const TSEOptions& tTseOpt, const std::string& tIgnoredParam)
{
	std::string method[3];
	method[ACCESS_BY_SERIAL_ESCPOS]		= "ACCESS_BY_SERIAL_ESCPOS";
	method[ACCESS_BY_NETWORK_ESCPOS]	= "ACCESS_BY_NETWORK_ESCPOS";
	method[ACCESS_BY_EPOS_DEVICE]		= "ACCESS_BY_EPOS_DEVICE";

	std::stringstream sLogAll;
	sLogAll << "VERSION: " << EPSONTSEDEMO_VERSION_S << std::endl;
	sLogAll << "[PARAMETERS SELECTED]" << std::endl;
	sLogAll << "Method: " << method[tTseOpt.mMethod] << std::endl;
	sLogAll << "Device ID: " << tTseOpt.mIdentifier << std::endl;
	sLogAll << "TSE Handle: " << tTseOpt.mTseHandler << std::endl << std::endl;

	if (tIgnoredParam != "")
	{
		sLogAll << "[IGNORED PARAMETERS]" << std::endl;
		sLogAll << tIgnoredParam;
	}

	TSEPRINTLOG(LIBLOG_LEVEL_INFO, ("%s", sLogAll.str().c_str()));
}

bool GetOptions(int tOptCount, char **tOptArg, TSEOptions& tTseOpt, TSELogOptions& tTseLog, std::string& tIgnoredParam)
{
	std::stringstream ignoredParam;
	if (tOptCount < 3)
	{
		return false;
	}

	for (int i = 1; i < tOptCount; i++)
	{
		if (tOptArg[i][0] != '-')
		{
			ignoredParam << tOptArg[i] << " is not a valid parameter." << std::endl;
			continue;
		}

		if (strlen(tOptArg[i]) <= 2)
		{
			ignoredParam << tOptArg[i] << " is not a valid parameter." << std::endl;
			continue;
		}

		switch (tOptArg[i][1])
		{
			case 'm':
			{
				unsigned short nMethod = tOptArg[i][2] - '0';
				if ((nMethod != ACCESS_BY_SERIAL_ESCPOS) &&
					(nMethod != ACCESS_BY_NETWORK_ESCPOS) &&
					(nMethod != ACCESS_BY_EPOS_DEVICE)) 
				{
					return false;
				}
				tTseOpt.mMethod = nMethod;
				break;
			}
			case 'i': tTseOpt.mIdentifier = (char *)(tOptArg[i] + 2);  break;
			case 't': tTseOpt.mTseHandler = (char *)(tOptArg[i] + 2);  break;
			
			case 'w': tTseLog.mLogLevel = tOptArg[i][2] - '0';  break;
			case 'x': tTseLog.mLogMethod |= ((tOptArg[i][2] == '1') ? (1<<0) : 0);  break;
			case 'y': tTseLog.mLogMethod |= ((tOptArg[i][2] == '1') ? (1<<1) : 0);  break;
			case 'z': tTseLog.mLogMethod |= ((tOptArg[i][2] == '1') ? (1<<2) : 0);  break;
			
			default:
			{
				ignoredParam << tOptArg[i] << " is not a valid parameter." << std::endl;
			}
		}
	}

	if (tTseOpt.mIdentifier == "") return false;

	tIgnoredParam = ignoredParam.str();
	return true;
}

int __cdecl main(int argc, char **argv)
{
	TSEOptions tseOpt;
	TSELogOptions tseLog;
	std::string ignoredParam = "";
	bool isShowMenu = true;
	std::stringstream sLogAll;
	
	std::cout << argv[0] << std::endl;
	std::cout << "VERSION: " << EPSONTSEDEMO_VERSION_S << std::endl;

	sLogAll << argv[0] << std::endl;
	sLogAll << "Version: " << EPSONTSEDEMO_VERSION_S << std::endl;
	sLogAll << EPSONTSEDEMO_WARNING_S << std::endl;
	sLogAll << std::endl;

	if (GetOptions(argc, argv, tseOpt, tseLog, ignoredParam) == false)
	{
		PrintUsage(argv[0]);
		return -1;
	}

	TSEPrintLog::SetLog(std::string(""), (LiblogLevel)tseLog.mLogLevel, (uint8_t)tseLog.mLogMethod, std::string(".//EpsonTseDemoExe.log"));
	TSEPRINTLOG(LIBLOG_LEVEL_INFO, ("%s",sLogAll.str().c_str()));
	ShowOptions(tseOpt, ignoredParam);

	while (isShowMenu)
	{
		CuiOperateTSE tseCui(tseOpt);
		if ((tseOpt.mTseHandler == "") && (tseOpt.mMethod == ACCESS_BY_EPOS_DEVICE))
		{
			if (tseCui.CuiGetTseId(tseOpt.mIdentifier, tseOpt.mTseHandler) != true)
			{
				break;
			}

			if (tseOpt.mTseHandler == "bye")
			{
				break;
			}
		}
		else
		{
			if (tseOpt.mTseHandler == "")
			{
				tseOpt.mTseHandler = "local_TSE";
			}
			if (tseCui.ConnectToTse(tseOpt) != true)
			{
				break;
			}
			isShowMenu = tseCui.ShowMenu();
			tseCui.DisconnectFromTse();
		}
	}
	return 0;
}

//EOF